# ovosolutions_backend/celery.py
import os
from celery import Celery
from celery.schedules import crontab
import logging

logger = logging.getLogger(__name__)

# Set default Django settings
os.environ.setdefault("DJANGO_SETTINGS_MODULE", "ovosolutions_backend.settings")

app = Celery("ovosolutions_backend")

# Load settings from Django config
app.config_from_object("django.conf:settings", namespace="CELERY")

# Auto-discover tasks from all installed apps
app.autodiscover_tasks()


# ---------------------------------------------------
# 🔹 CELERY BEAT SCHEDULE (every 12 hours)
# ---------------------------------------------------
app.conf.beat_schedule = {
    "auto-sync-frontend-data": {
        "task": "website.tasks.sync_frontend_data",
        # Run every 12 hours (midnight + noon)
        "schedule": crontab(hour="0,12", minute=0),
        "options": {"queue": "default"},
    },
}

# Optional: Celery timezone
app.conf.timezone = "Africa/Nairobi"

# Optional: retry / safety configs
app.conf.update(
    task_acks_late=True,
    task_reject_on_worker_lost=True,
    worker_prefetch_multiplier=1,
    broker_connection_retry_on_startup=True,
)


# ---------------------------------------------------
# 🔹 Diagnostic Logging
# ---------------------------------------------------
@app.on_after_configure.connect
def announce_start(sender, **kwargs):
    logger.info("✅ Celery app configured with beat schedule (every 12h).")


@app.task(bind=True)
def debug_task(self):
    logger.info(f"Celery Debug Task executed: {self.request!r}")
