# ovosolutions_backend/settings.py - PRODUCTION OPTIMIZED
import os
import sys
from pathlib import Path
from dotenv import load_dotenv

# =====================================================
# 🌍 ENVIRONMENT SETUP - ENHANCED
# =====================================================
BASE_DIR = Path(__file__).resolve().parent.parent

# Load environment variables with fallbacks
env_path = BASE_DIR / ".env"
if env_path.exists():
    load_dotenv(env_path)
    print(f"✅ Loaded environment from {env_path}")
else:
    print(f"⚠️  .env file not found at {env_path}")

# Environment detection with better defaults
ENVIRONMENT = os.getenv("DJANGO_ENV", "production").lower()
IS_PRODUCTION = ENVIRONMENT == "production"
IS_DEVELOPMENT = ENVIRONMENT == "development"
IS_TESTING = "test" in sys.argv or ENVIRONMENT == "testing"

print(f"🚀 Starting Django in {ENVIRONMENT.upper()} mode")

# =====================================================
# 🔐 SECURITY CONFIGURATION - BULLETPROOF
# =====================================================
# Enhanced SECRET_KEY handling with multiple fallbacks
SECRET_KEY = os.getenv("DJANGO_SECRET_KEY") or os.getenv("SECRET_KEY")

if not SECRET_KEY:
    if IS_PRODUCTION:
        # Try to load from cPanel environment variables
        try:
            SECRET_KEY = os.environ.get('SECRET_KEY')
            if not SECRET_KEY:
                raise Exception(
                    "❌ SECRET_KEY must be set in production!\n"
                    "Set it in your .env file or cPanel environment variables:\n"
                    "1. Create .env file in project root with: SECRET_KEY=your-key\n"
                    "2. Or set in cPanel: Python App → Environment Variables\n"
                    f"Current path: {BASE_DIR}\n"
                    f"Env file exists: {env_path.exists()}"
                )
        except:
            raise
    else:
        SECRET_KEY = "dev-key-change-this-before-production-" + ENVIRONMENT
        print("⚠️  Using development SECRET_KEY - CHANGE IN PRODUCTION!")

# Security settings - Smart production defaults
DEBUG = os.getenv("DJANGO_DEBUG", "0") == "1"

# Production security settings (conditional for easier debugging)
if IS_PRODUCTION and not DEBUG:
    SECURE_SSL_REDIRECT = True
    SECURE_HSTS_SECONDS = 31536000  # 1 year
    SECURE_HSTS_INCLUDE_SUBDOMAINS = True
    SECURE_HSTS_PRELOAD = True
    SESSION_COOKIE_SECURE = True
    CSRF_COOKIE_SECURE = True
    SECURE_PROXY_SSL_HEADER = ('HTTP_X_FORWARDED_PROTO', 'https')
else:
    # Relaxed for development/testing
    SECURE_SSL_REDIRECT = False
    SECURE_HSTS_SECONDS = 0
    SESSION_COOKIE_SECURE = False
    CSRF_COOKIE_SECURE = False

# Always enabled security headers
SECURE_BROWSER_XSS_FILTER = True
SECURE_CONTENT_TYPE_NOSNIFF = True
X_FRAME_OPTIONS = "DENY"

# Enhanced ALLOWED_HOSTS with IP detection
ALLOWED_HOSTS = []

# Add hosts from environment
env_hosts = os.getenv("ALLOWED_HOSTS", "")
if env_hosts:
    ALLOWED_HOSTS.extend([host.strip() for host in env_hosts.split(",") if host.strip()])

# Add default production hosts
if IS_PRODUCTION:
    ALLOWED_HOSTS.extend([
        "seovosolutions.com",
        "www.seovosolutions.com",
        ".seovosolutions.com",  # All subdomains
    ])

# Development defaults
if IS_DEVELOPMENT:
    ALLOWED_HOSTS.extend([
        "localhost",
        "127.0.0.1",
        "0.0.0.0",
        "::1",
        "testserver",
    ])
    if DEBUG:
        ALLOWED_HOSTS.append("*")  # Allow all in debug mode

# Ensure we have at least one host
if not ALLOWED_HOSTS:
    ALLOWED_HOSTS = ["localhost", "127.0.0.1"]

# =====================================================
# 🧩 APPLICATION DEFINITION - OPTIMIZED
# =====================================================
INSTALLED_APPS = [
    # Django Core
    "django.contrib.admin",
    "django.contrib.auth",
    "django.contrib.contenttypes",
    "django.contrib.sessions",
    "django.contrib.messages",
    "django.contrib.staticfiles",

    # Third-Party (Production Essential)
    "rest_framework",
    "rest_framework.authtoken",
    "corsheaders",
    "whitenoise.runserver_nostatic",
    
    # Local Apps
    "website",
]

# =====================================================
# ⚙️ MIDDLEWARE CONFIGURATION - OPTIMIZED
# =====================================================
MIDDLEWARE = [
    # Security and CORS
    "corsheaders.middleware.CorsMiddleware",
    "django.middleware.security.SecurityMiddleware",
    
    # Whitenoise for static files (must be after SecurityMiddleware)
    "whitenoise.middleware.WhiteNoiseMiddleware",
    
    # Django core
    "django.contrib.sessions.middleware.SessionMiddleware",
    "django.middleware.common.CommonMiddleware",
    "django.middleware.csrf.CsrfViewMiddleware",
    "django.contrib.auth.middleware.AuthenticationMiddleware",
    "django.contrib.messages.middleware.MessageMiddleware",
    "django.middleware.clickjacking.XFrameOptionsMiddleware",
]

# =====================================================
# 🌐 URLS & WSGI CONFIGURATION
# =====================================================
ROOT_URLCONF = "ovosolutions_backend.urls"
WSGI_APPLICATION = "ovosolutions_backend.wsgi.application"

# =====================================================
# 🗃 DATABASE CONFIGURATION - PRODUCTION READY
# =====================================================
# Try MySQL first, fallback to SQLite for development
DATABASES = {}

# Database configuration with proper error handling
try:
    if IS_PRODUCTION or os.getenv("USE_MYSQL", "1") == "1":
        db_password = os.getenv("DB_PASSWORD") or os.getenv("SeovoSolutions251")
        
        if not db_password and IS_PRODUCTION:
            print("⚠️  Database password not set. Check your .env file")
            db_password = ""  # Will fail with proper error
        
        DATABASES["default"] = {
            "ENGINE": "django.db.backends.mysql",
            "NAME": os.getenv("DB_NAME", "ehictriq_SeovoSolutions"),
            "USER": os.getenv("DB_USER", "ehictriq_Seovo"),
            "PASSWORD": db_password,
            "HOST": os.getenv("DB_HOST", "localhost"),
            "PORT": os.getenv("DB_PORT", "3306"),
            "OPTIONS": {
                'init_command': "SET sql_mode='STRICT_TRANS_TABLES'",
                'charset': 'utf8mb4',
                'connect_timeout': 10,
            },
            'CONN_MAX_AGE': 300 if IS_PRODUCTION else 0,
        }
        print("✅ Using MySQL database configuration")
    else:
        raise Exception("Using SQLite for development")
        
except Exception as e:
    if IS_DEVELOPMENT or DEBUG:
        DATABASES["default"] = {
            "ENGINE": "django.db.backends.sqlite3",
            "NAME": BASE_DIR / "db.sqlite3",
        }
        print(f"⚠️  Using SQLite for development: {e}")
    else:
        print(f"❌ Database configuration error: {e}")
        raise

# =====================================================
# 🧱 STATIC & MEDIA FILES - CPANEL OPTIMIZED
# =====================================================
STATIC_URL = "/static/"
STATIC_ROOT = BASE_DIR / "staticfiles"
STATICFILES_DIRS = [BASE_DIR / "static"]

MEDIA_URL = "/media/"
MEDIA_ROOT = BASE_DIR / "media"

# Create directories with proper permissions
for directory in [STATIC_ROOT, MEDIA_ROOT]:
    os.makedirs(directory, exist_ok=True)

# Whitenoise configuration for production
if IS_PRODUCTION:
    STATICFILES_STORAGE = "whitenoise.storage.CompressedManifestStaticFilesStorage"
    WHITENOISE_ROOT = STATIC_ROOT
    WHITENOISE_AUTOREFRESH = False
    WHITENOISE_MAX_AGE = 31536000  # 1 year cache
    WHITENOISE_INDEX_FILE = True
else:
    STATICFILES_STORAGE = "django.contrib.staticfiles.storage.StaticFilesStorage"

# =====================================================
# 🔒 CORS & CSRF CONFIGURATION - SECURE
# =====================================================
# CORS Configuration
CORS_ALLOW_ALL_ORIGINS = False  # Never allow all in production

# Production CORS origins
CORS_ALLOWED_ORIGINS = [
    "https://seovosolutions.com",
    "https://www.seovosolutions.com",
]

# Add from environment
env_cors = os.getenv("CORS_ALLOWED_ORIGINS", "")
if env_cors:
    CORS_ALLOWED_ORIGINS.extend([
        origin.strip() for origin in env_cors.split(",") if origin.strip()
    ])

# Development CORS
if IS_DEVELOPMENT:
    CORS_ALLOWED_ORIGINS.extend([
        "http://localhost:3000",
        "http://127.0.0.1:3000",
        "http://localhost:8000",
        "http://127.0.0.1:8000",
    ])
    if DEBUG:
        CORS_ALLOW_ALL_ORIGINS = True

CORS_ALLOW_CREDENTIALS = True
CORS_ALLOW_METHODS = [
    "DELETE",
    "GET",
    "OPTIONS",
    "PATCH",
    "POST",
    "PUT",
]
CORS_ALLOW_HEADERS = [
    "accept",
    "accept-encoding",
    "authorization",
    "content-type",
    "dnt",
    "origin",
    "user-agent",
    "x-csrftoken",
    "x-requested-with",
]

# CSRF Configuration
CSRF_TRUSTED_ORIGINS = CORS_ALLOWED_ORIGINS.copy()

# Add environment CSRF origins
env_csrf = os.getenv("CSRF_TRUSTED_ORIGINS", "")
if env_csrf:
    CSRF_TRUSTED_ORIGINS.extend([
        origin.strip() for origin in env_csrf.split(",") if origin.strip()
    ])

# =====================================================
# 🔐 REST FRAMEWORK CONFIGURATION - PRODUCTION
# =====================================================
REST_FRAMEWORK = {
    "DEFAULT_AUTHENTICATION_CLASSES": [
        "rest_framework.authentication.SessionAuthentication",
        "rest_framework.authentication.TokenAuthentication",
    ],
    "DEFAULT_PERMISSION_CLASSES": [
        "rest_framework.permissions.IsAuthenticatedOrReadOnly",
    ],
    "DEFAULT_RENDERER_CLASSES": [
        "rest_framework.renderers.JSONRenderer",
    ],
    "DEFAULT_PARSER_CLASSES": [
        "rest_framework.parsers.JSONParser",
        "rest_framework.parsers.MultiPartParser",
        "rest_framework.parsers.FormParser",
    ],
    "DEFAULT_THROTTLE_CLASSES": [
        "rest_framework.throttling.AnonRateThrottle",
        "rest_framework.throttling.UserRateThrottle"
    ],
    "DEFAULT_THROTTLE_RATES": {
        "anon": "100/hour",
        "user": "1000/hour"
    },
    "DEFAULT_PAGINATION_CLASS": "rest_framework.pagination.PageNumberPagination",
    "PAGE_SIZE": 20,
    "DEFAULT_SCHEMA_CLASS": "rest_framework.schemas.coreapi.AutoSchema",
    "EXCEPTION_HANDLER": "rest_framework.views.exception_handler",
}

# Add browsable API only in development
if IS_DEVELOPMENT and DEBUG:
    REST_FRAMEWORK["DEFAULT_RENDERER_CLASSES"].append("rest_framework.renderers.BrowsableAPIRenderer")
    REST_FRAMEWORK["DEFAULT_PERMISSION_CLASSES"] = ["rest_framework.permissions.AllowAny"]

# =====================================================
# ⚡ CACHING CONFIGURATION - PRODUCTION OPTIMIZED
# =====================================================
# File-based cache for cPanel (reliable and persistent)
CACHE_DIR = BASE_DIR / "cache"
os.makedirs(CACHE_DIR, exist_ok=True)

CACHES = {
    "default": {
        "BACKEND": "django.core.cache.backends.filebased.FileBasedCache",
        "LOCATION": CACHE_DIR,
        "TIMEOUT": 300,  # 5 minutes
        "OPTIONS": {
            "MAX_ENTRIES": 10000
        }
    }
}

# Session configuration
SESSION_ENGINE = "django.contrib.sessions.backends.cached_db"
SESSION_CACHE_ALIAS = "default"
SESSION_COOKIE_AGE = 1209600  # 2 weeks in seconds
SESSION_SAVE_EVERY_REQUEST = True

# =====================================================
# 📧 EMAIL CONFIGURATION - PRODUCTION READY
# =====================================================
EMAIL_BACKEND = "django.core.mail.backends.smtp.EmailBackend"
EMAIL_HOST = os.getenv("EMAIL_HOST", "smtp.gmail.com")
EMAIL_PORT = int(os.getenv("EMAIL_PORT", "587"))
EMAIL_USE_TLS = os.getenv("EMAIL_USE_TLS", "True") == "True"
EMAIL_HOST_USER = os.getenv("EMAIL_HOST_USER", "")
EMAIL_HOST_PASSWORD = os.getenv("EMAIL_HOST_PASSWORD", "")
EMAIL_TIMEOUT = 30

DEFAULT_FROM_EMAIL = os.getenv("DEFAULT_FROM_EMAIL", "noreply@seovosolutions.com")
SERVER_EMAIL = DEFAULT_FROM_EMAIL
ADMINS = [
    ("Admin", os.getenv("ADMIN_EMAIL", "admin@seovosolutions.com")),
]

# Console backend for development
if IS_DEVELOPMENT and not IS_TESTING:
    EMAIL_BACKEND = "django.core.mail.backends.console.EmailBackend"
    print("📧 Using console email backend for development")

# =====================================================
# 🧾 LOGGING CONFIGURATION - PRODUCTION GRADE
# =====================================================
LOG_DIR = BASE_DIR / "logs"
os.makedirs(LOG_DIR, exist_ok=True)

LOGGING = {
    "version": 1,
    "disable_existing_loggers": False,
    "formatters": {
        "verbose": {
            "format": "[{asctime}] {levelname} {name}:{lineno} {message}",
            "style": "{",
            "datefmt": "%Y-%m-%d %H:%M:%S"
        },
        "simple": {
            "format": "{levelname} {message}",
            "style": "{",
        },
    },
    "handlers": {
        "console": {
            "class": "logging.StreamHandler",
            "formatter": "simple",
        },
        "file": {
            "class": "logging.handlers.RotatingFileHandler",
            "filename": LOG_DIR / "django.log",
            "maxBytes": 10485760,  # 10MB
            "backupCount": 10,
            "formatter": "verbose",
        },
        "error_file": {
            "class": "logging.handlers.RotatingFileHandler",
            "filename": LOG_DIR / "errors.log",
            "maxBytes": 10485760,
            "backupCount": 10,
            "formatter": "verbose",
            "level": "ERROR",
        },
    },
    "root": {
        "handlers": ["console", "file", "error_file"] if IS_PRODUCTION else ["console"],
        "level": "INFO" if IS_PRODUCTION else "DEBUG",
    },
    "loggers": {
        "django": {
            "handlers": ["console", "file"],
            "level": "INFO",
            "propagate": False,
        },
        "django.request": {
            "handlers": ["console", "error_file"],
            "level": "ERROR",
            "propagate": False,
        },
        "django.db.backends": {
            "handlers": ["console"],
            "level": "WARNING",
            "propagate": False,
        },
        "website": {
            "handlers": ["console", "file"],
            "level": "DEBUG" if IS_DEVELOPMENT else "INFO",
            "propagate": False,
        },
    },
}

# =====================================================
# 🧠 TEMPLATE CONFIGURATION
# =====================================================
TEMPLATES = [
    {
        "BACKEND": "django.template.backends.django.DjangoTemplates",
        "DIRS": [BASE_DIR / "templates"],
        "APP_DIRS": True,
        "OPTIONS": {
            "context_processors": [
                "django.template.context_processors.debug",
                "django.template.context_processors.request",
                "django.contrib.auth.context_processors.auth",
                "django.contrib.messages.context_processors.messages",
                "website.context_processors.site_config",  # Custom context processor
            ],
            "debug": DEBUG,
        },
    },
]

# =====================================================
# 🔧 INTERNATIONALIZATION
# =====================================================
LANGUAGE_CODE = "en-us"
TIME_ZONE = os.getenv("TIME_ZONE", "Africa/Nairobi")
USE_I18N = True
USE_TZ = True
USE_L10N = True

# =====================================================
# 🔑 AUTHENTICATION & PASSWORDS
# =====================================================
AUTH_PASSWORD_VALIDATORS = [
    {
        "NAME": "django.contrib.auth.password_validation.UserAttributeSimilarityValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation.MinimumLengthValidator",
        "OPTIONS": {"min_length": 8}
    },
    {
        "NAME": "django.contrib.auth.password_validation.CommonPasswordValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation.NumericPasswordValidator",
    },
]

# Password hashers (production optimized)
PASSWORD_HASHERS = [
    "django.contrib.auth.hashers.Argon2PasswordHasher",
    "django.contrib.auth.hashers.PBKDF2PasswordHasher",
    "django.contrib.auth.hashers.PBKDF2SHA1PasswordHasher",
    "django.contrib.auth.hashers.BCryptSHA256PasswordHasher",
]

# =====================================================
# ⚙️ APPLICATION SPECIFIC SETTINGS
# =====================================================
# Site configuration
SITE_NAME = "SeovoSolutions"
SITE_DOMAIN = "seovosolutions.com"
API_VERSION = "2.0.0"
MAX_FILE_UPLOAD_SIZE = 25 * 1024 * 1024  # 25MB

# Frontend configuration
FRONTEND_URL = os.getenv("FRONTEND_URL", "https://www.seovosolutions.com")

# AI Configuration (optional)
GEMINI_API_KEY = os.getenv("GEMINI_API_KEY", "")
OPENAI_API_KEY = os.getenv("OPENAI_API_KEY", "")

# =====================================================
# 🚀 FINAL VALIDATION & SETUP
# =====================================================
def validate_settings():
    """Validate critical settings"""
    errors = []
    
    # Check SECRET_KEY
    if SECRET_KEY.startswith("dev-key-") and IS_PRODUCTION:
        errors.append("SECRET_KEY is using development default in production!")
    
    # Check database
    if IS_PRODUCTION and DATABASES["default"]["ENGINE"] == "django.db.backends.sqlite3":
        errors.append("Using SQLite in production is not recommended!")
    
    # Check DEBUG mode
    if IS_PRODUCTION and DEBUG:
        errors.append("DEBUG is True in production!")
    
    # Check ALLOWED_HOSTS
    if IS_PRODUCTION and ("*" in ALLOWED_HOSTS or "localhost" in ALLOWED_HOSTS):
        errors.append("ALLOWED_HOSTS contains insecure values in production!")
    
    if errors:
        print("\n❌ SETTINGS VALIDATION ERRORS:")
        for error in errors:
            print(f"   - {error}")
        if IS_PRODUCTION:
            print("\n⚠️  FIX THESE ISSUES BEFORE DEPLOYING TO PRODUCTION!\n")
    
    return len(errors) == 0

# Run validation
settings_valid = validate_settings()

# Final startup message
print(f"\n{'='*60}")
print(f"🚀 SEOVOSOLUTIONS BACKEND - {ENVIRONMENT.upper()} MODE")
print(f"{'='*60}")
print(f"✅ Django Version: {__import__('django').__version__}")
print(f"✅ Debug Mode: {DEBUG}")
print(f"✅ Environment: {ENVIRONMENT}")
print(f"✅ Database: {DATABASES['default']['ENGINE']}")
print(f"✅ Static Files: {STATIC_URL} -> {STATIC_ROOT}")
print(f"✅ Media Files: {MEDIA_URL} -> {MEDIA_ROOT}")
print(f"✅ Allowed Hosts: {len(ALLOWED_HOSTS)} configured")
print(f"✅ CORS Origins: {len(CORS_ALLOWED_ORIGINS)} configured")
print(f"✅ Timezone: {TIME_ZONE}")
print(f"✅ Settings Validation: {'PASSED' if settings_valid else 'FAILED'}")
print(f"{'='*60}\n")

if not settings_valid and IS_PRODUCTION:
    print("⚠️  WARNING: Some production settings need attention!")
    print("   Please review the validation errors above.\n")