from django.contrib import admin
from django.utils import timezone
from .models import (
    Contact,
    TrainingProgram,
    TrainingRegistration,
    FileUpload,
    Post,
    Event,
    Testimonial,
    Notification,
    Document,
    ConversationLog,
    AssistantFeedback,
    SiteConfig,
    Category,
    BlovionPost,
    SystemMetric,
    PageView,
)

# =====================================================================
# 🔹 GENERIC MIXINS
# =====================================================================
class AuditFields(admin.ModelAdmin):
    """Common admin settings: readonly audit fields."""
    readonly_fields = ("created_at", "updated_at")
    ordering = ("-created_at",)
    list_per_page = 25


# =====================================================================
# 🔹 SITE CONFIGURATION & ANALYTICS
# =====================================================================
@admin.register(SiteConfig)
class SiteConfigAdmin(admin.ModelAdmin):
    list_display = ("name", "system_status", "last_sync", "maintenance_mode", "version")
    readonly_fields = ("last_sync",)
    actions = ["toggle_maintenance", "update_sync_time"]

    @admin.action(description="Toggle maintenance mode")
    def toggle_maintenance(self, request, queryset):
        for config in queryset:
            config.maintenance_mode = not config.maintenance_mode
            config.save()
        self.message_user(request, "Maintenance mode toggled for selected configs.")

    @admin.action(description="Update last sync timestamp")
    def update_sync_time(self, request, queryset):
        now = timezone.now()
        updated = queryset.update(last_sync=now)
        self.message_user(request, f"{updated} configuration(s) updated to {now:%Y-%m-%d %H:%M:%S}.")


@admin.register(SystemMetric)
class SystemMetricAdmin(admin.ModelAdmin):  # ✅ FIXED: Don't inherit from AuditFields
    list_display = ("metric_type", "value", "recorded_at")
    list_filter = ("metric_type", "recorded_at")
    search_fields = ("metric_type",)
    readonly_fields = ("recorded_at",)
    ordering = ("-recorded_at",)  # ✅ FIXED: Use recorded_at instead of created_at
    list_per_page = 25


@admin.register(PageView)
class PageViewAdmin(AuditFields):
    list_display = ("page_url", "user", "created_at", "duration")
    list_filter = ("created_at",)
    search_fields = ("page_url", "page_title", "user__username")
    readonly_fields = ("created_at",)


# =====================================================================
# 🔹 CATEGORIES
# =====================================================================
@admin.register(Category)
class CategoryAdmin(AuditFields):
    list_display = ("name", "slug", "is_active", "post_count", "order", "created_at")
    list_filter = ("is_active", "created_at")
    search_fields = ("name", "description")
    prepopulated_fields = {"slug": ("name",)}
    readonly_fields = ("created_at", "updated_at", "post_count")
    actions = ["update_post_counts"]

    @admin.action(description="Update post counts for selected categories")
    def update_post_counts(self, request, queryset):
        for category in queryset:
            category.update_post_count()
        self.message_user(request, "Post counts updated for selected categories.")


# =====================================================================
# 🔹 CONTACTS
# =====================================================================
@admin.register(Contact)
class ContactAdmin(AuditFields):
    list_display = ("name", "email", "contact_type", "subject", "created_at", "responded", "priority")
    list_filter = ("responded", "contact_type", "priority", "created_at")
    search_fields = ("name", "email", "subject", "message")
    readonly_fields = ("created_at", "updated_at")
    actions = ["mark_as_responded", "mark_as_high_priority"]

    @admin.action(description="Mark selected contacts as responded")
    def mark_as_responded(self, request, queryset):
        updated = queryset.update(responded=True, responded_at=timezone.now())
        self.message_user(request, f"{updated} messages marked as responded.")

    @admin.action(description="Mark selected as high priority")
    def mark_as_high_priority(self, request, queryset):
        updated = queryset.update(priority="high")
        self.message_user(request, f"{updated} contacts marked as high priority.")


# =====================================================================
# 🔹 TRAINING PROGRAMS & REGISTRATIONS
# =====================================================================
class TrainingRegistrationInline(admin.TabularInline):
    model = TrainingRegistration
    extra = 0
    readonly_fields = ("created_at", "registration_code")
    fields = ("full_name", "email", "status", "payment_status", "created_at")


@admin.register(TrainingProgram)
class TrainingProgramAdmin(AuditFields):
    list_display = ("title", "program_type", "level", "start_date", "end_date", "capacity", "current_enrollment", "registration_open", "is_active")
    list_filter = ("program_type", "level", "registration_open", "is_active", "start_date")
    search_fields = ("title", "instructor", "description", "short_description")
    inlines = [TrainingRegistrationInline]
    prepopulated_fields = {"slug": ("title",)}
    readonly_fields = ("created_at", "updated_at", "current_enrollment", "view_count")
    actions = ["open_registration", "close_registration"]

    @admin.action(description="Open registration for selected programs")
    def open_registration(self, request, queryset):
        updated = queryset.update(registration_open=True)
        self.message_user(request, f"{updated} program(s) registration opened.")

    @admin.action(description="Close registration for selected programs")
    def close_registration(self, request, queryset):
        updated = queryset.update(registration_open=False)
        self.message_user(request, f"{updated} program(s) registration closed.")


@admin.register(TrainingRegistration)
class TrainingRegistrationAdmin(AuditFields):
    list_display = ("full_name", "email", "program", "status", "payment_status", "amount_paid", "created_at")
    list_filter = ("status", "payment_status", "program", "created_at")
    search_fields = ("full_name", "email", "program__title", "registration_code")
    readonly_fields = ("created_at", "updated_at", "registration_code")
    actions = ["confirm_registrations", "mark_as_paid"]

    @admin.action(description="Confirm selected registrations")
    def confirm_registrations(self, request, queryset):
        updated = queryset.update(status="confirmed")
        self.message_user(request, f"{updated} registration(s) confirmed.")

    @admin.action(description="Mark selected as paid")
    def mark_as_paid(self, request, queryset):
        updated = queryset.update(payment_status="paid")
        self.message_user(request, f"{updated} registration(s) marked as paid.")


# =====================================================================
# 🔹 FILES & MEDIA
# =====================================================================
@admin.register(FileUpload)
class FileUploadAdmin(AuditFields):
    list_display = ("title", "file_name", "category", "file_type", "size_kb", "uploaded_by", "is_public", "created_at")
    list_filter = ("category", "is_public", "created_at")
    search_fields = ("title", "file_name", "description")
    readonly_fields = ("created_at", "updated_at", "size_kb", "file_size", "file_type", "access_count")
    actions = ["make_public", "make_private"]

    @admin.action(description="Make selected files public")
    def make_public(self, request, queryset):
        updated = queryset.update(is_public=True)
        self.message_user(request, f"{updated} file(s) made public.")

    @admin.action(description="Make selected files private")
    def make_private(self, request, queryset):
        updated = queryset.update(is_public=False)
        self.message_user(request, f"{updated} file(s) made private.")


# =====================================================================
# 🔹 POSTS / BLOGS
# =====================================================================
@admin.register(Post)
class PostAdmin(AuditFields):
    list_display = ("title", "author", "post_type", "status", "published", "featured", "view_count", "created_at")
    list_filter = ("post_type", "status", "published", "featured", "created_at")
    search_fields = ("title", "content", "excerpt", "author__username")
    prepopulated_fields = {"slug": ("title",)}
    readonly_fields = ("created_at", "updated_at", "view_count", "last_viewed_at", "word_count")
    filter_horizontal = ("categories", "attachments")
    actions = ["publish_posts", "unpublish_posts", "feature_posts"]

    @admin.action(description="Publish selected posts")
    def publish_posts(self, request, queryset):
        updated = queryset.update(status="published", published=True, published_at=timezone.now())
        self.message_user(request, f"{updated} post(s) published successfully.")

    @admin.action(description="Unpublish selected posts")
    def unpublish_posts(self, request, queryset):
        updated = queryset.update(status="draft", published=False)
        self.message_user(request, f"{updated} post(s) unpublished.")

    @admin.action(description="Feature selected posts")
    def feature_posts(self, request, queryset):
        updated = queryset.update(featured=True)
        self.message_user(request, f"{updated} post(s) featured.")


# =====================================================================
# 🔹 BLOVION POSTS
# =====================================================================
@admin.register(BlovionPost)
class BlovionPostAdmin(AuditFields):
    list_display = ("title", "post_type", "status", "published", "featured", "author", "view_count", "created_at")
    list_filter = ("post_type", "status", "published", "featured", "complexity", "created_at")
    search_fields = ("title", "caption", "content", "author")
    prepopulated_fields = {"slug": ("title",)}
    readonly_fields = ("created_at", "updated_at", "view_count", "last_viewed_at", "word_count")
    filter_horizontal = ("categories",)
    actions = ["publish_posts", "feature_posts", "mark_as_social_posted"]

    @admin.action(description="Publish selected Blovion posts")
    def publish_posts(self, request, queryset):
        updated = queryset.update(status="published", published=True, published_at=timezone.now())
        self.message_user(request, f"{updated} Blovion post(s) published successfully.")

    @admin.action(description="Feature selected Blovion posts")
    def feature_posts(self, request, queryset):
        updated = queryset.update(featured=True)
        self.message_user(request, f"{updated} Blovion post(s) featured.")

    @admin.action(description="Mark as social posted")
    def mark_as_social_posted(self, request, queryset):
        updated = queryset.update(social_posted=True)
        self.message_user(request, f"{updated} Blovion post(s) marked as social posted.")


# =====================================================================
# 🔹 EVENTS
# =====================================================================
@admin.register(Event)
class EventAdmin(AuditFields):
    list_display = ("title", "event_type", "start_time", "end_time", "location", "is_virtual", "capacity", "current_registrations", "registration_open")
    list_filter = ("event_type", "is_virtual", "registration_open", "featured", "start_time")
    search_fields = ("title", "description", "location", "organizer__username")
    readonly_fields = ("created_at", "updated_at", "current_registrations", "view_count")
    actions = ["open_registration", "close_registration", "feature_events"]

    @admin.action(description="Open registration for selected events")
    def open_registration(self, request, queryset):
        updated = queryset.update(registration_open=True)
        self.message_user(request, f"{updated} event(s) registration opened.")

    @admin.action(description="Close registration for selected events")
    def close_registration(self, request, queryset):
        updated = queryset.update(registration_open=False)
        self.message_user(request, f"{updated} event(s) registration closed.")

    @admin.action(description="Feature selected events")
    def feature_events(self, request, queryset):
        updated = queryset.update(featured=True)
        self.message_user(request, f"{updated} event(s) featured.")


# =====================================================================
# 🔹 TESTIMONIALS
# =====================================================================
@admin.register(Testimonial)
class TestimonialAdmin(AuditFields):
    list_display = ("name", "company", "rating", "approved", "featured", "display_order", "created_at")
    list_filter = ("approved", "featured", "rating", "created_at")
    search_fields = ("name", "company", "message", "project_type")
    readonly_fields = ("created_at", "updated_at")
    actions = ["approve_testimonials", "feature_testimonials"]

    @admin.action(description="Approve selected testimonials")
    def approve_testimonials(self, request, queryset):
        updated = queryset.update(approved=True)
        self.message_user(request, f"{updated} testimonials approved.")

    @admin.action(description="Feature selected testimonials")
    def feature_testimonials(self, request, queryset):
        updated = queryset.update(featured=True)
        self.message_user(request, f"{updated} testimonials featured.")


# =====================================================================
# 🔹 NOTIFICATIONS
# =====================================================================
@admin.register(Notification)
class NotificationAdmin(AuditFields):
    list_display = ("type", "title", "recipient", "priority", "read", "created_at")
    list_filter = ("type", "read", "priority", "created_at")
    search_fields = ("title", "message", "recipient__username")
    readonly_fields = ("created_at", "updated_at")
    actions = ["mark_as_read", "mark_as_unread"]

    @admin.action(description="Mark selected notifications as read")
    def mark_as_read(self, request, queryset):
        updated = queryset.update(read=True, read_at=timezone.now())
        self.message_user(request, f"{updated} notifications marked as read.")

    @admin.action(description="Mark selected notifications as unread")
    def mark_as_unread(self, request, queryset):
        updated = queryset.update(read=False, read_at=None)
        self.message_user(request, f"{updated} notifications marked as unread.")


# =====================================================================
# 🔹 AI / ASSISTANT MODELS
# =====================================================================
@admin.register(Document)
class DocumentAdmin(AuditFields):
    list_display = ("title", "source", "source_id", "chunk_index", "total_chunks", "embedding_dim", "is_active", "created_at")
    list_filter = ("source", "is_active", "created_at")
    search_fields = ("title", "text", "source_id")
    readonly_fields = ("created_at", "updated_at")


@admin.register(ConversationLog)
class ConversationLogAdmin(AuditFields):
    list_display = ("user_identifier", "intent", "confidence", "user_satisfaction", "response_time", "created_at")
    list_filter = ("intent", "created_at")
    search_fields = ("user_identifier", "user_message", "bot_reply", "intent")
    readonly_fields = ("created_at", "updated_at")


@admin.register(AssistantFeedback)
class AssistantFeedbackAdmin(AuditFields):
    list_display = ("conversation", "helpful", "rating", "would_use_again", "created_at")
    list_filter = ("helpful", "would_use_again", "created_at")
    search_fields = ("comment", "improvements_suggested", "conversation__user_identifier")
    readonly_fields = ("created_at", "updated_at")
    
