import logging
from django.apps import AppConfig

logger = logging.getLogger(__name__)

class WebsiteConfig(AppConfig):
    default_auto_field = "django.db.models.BigAutoField"
    name = "website"
    verbose_name = "OvoSolutions Website Core"

    def ready(self):
        """
        Executed when the Django app is fully loaded.
        Responsibilities:
        - Register signals
        - Initialize optional background schedulers
        - Log app startup status safely (no encoding issues)
        """
        # ---------------------------
        # Load signals safely
        # ---------------------------
        try:
            import website.signals  # noqa: F401
            logger.info("Website signals loaded successfully.")
        except ImportError as e:
            logger.warning(f"Could not import website.signals: {e}")
        except Exception as e:
            logger.error(f"Unexpected error while loading signals: {e}", exc_info=True)

        # ---------------------------
        # Initialize background scheduler (optional)
        # ---------------------------
        try:
            from website.utils import start_auto_sync  # noqa: F401
            start_auto_sync()
            logger.info("Background auto-sync scheduler started successfully.")
        except ImportError:
            # The utils or scheduler may not exist in all environments
            logger.debug("No background auto-sync initialized (optional, utils not found).")
        except Exception as e:
            logger.error(f"Error starting background scheduler: {e}", exc_info=True)

        # ---------------------------
        # Final initialization log
        # ---------------------------
        try:
            logger.info("WebsiteConfig is ready and the app is fully initialized.")
        except Exception:
            # Prevent Windows console Unicode issues
            print("WebsiteConfig initialized successfully (logging safe mode).")
