from django.db import models
from django.contrib.auth import get_user_model
from django.utils import timezone
from django.utils.text import slugify
from django.core.validators import MinValueValidator, MaxValueValidator
from django.conf import settings
from django.urls import reverse
from django.core.exceptions import ValidationError
import random
import string
import os
import json

User = get_user_model()


# =====================================================
# 🔹 BASE MODELS (Enhanced with Frontend Support)
# =====================================================
class TimeStampedModel(models.Model):
    """Abstract base for created/updated timestamps with indexing."""
    created_at = models.DateTimeField(default=timezone.now, db_index=True)
    updated_at = models.DateTimeField(auto_now=True, db_index=True)

    class Meta:
        abstract = True
        ordering = ['-created_at']


class SoftDeletableModel(models.Model):
    """Adds soft deletion with indexing."""
    is_active = models.BooleanField(default=True, db_index=True)

    class Meta:
        abstract = True


class MetaDataModel(models.Model):
    """Reusable SEO and metadata fields with improved structure."""
    meta_title = models.CharField(max_length=255, blank=True, default='')
    meta_description = models.TextField(blank=True, default='', max_length=500)
    meta_keywords = models.CharField(max_length=500, blank=True, default='')
    tags = models.JSONField(default=list, blank=True)
    view_count = models.PositiveIntegerField(default=0, db_index=True)
    last_viewed_at = models.DateTimeField(null=True, blank=True)

    class Meta:
        abstract = True

    def increment_view_count(self):
        """Atomically increment view count."""
        self.view_count = models.F('view_count') + 1
        self.last_viewed_at = timezone.now()
        self.save(update_fields=['view_count', 'last_viewed_at'])

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'meta_title': self.meta_title,
            'meta_description': self.meta_description,
            'meta_keywords': self.meta_keywords,
            'tags': self.tags,
            'view_count': self.view_count,
            'last_viewed_at': self.last_viewed_at.isoformat() if self.last_viewed_at else None,
        }


class ArchivableModel(models.Model):
    """Adds archiving functionality with indexing."""
    archived = models.BooleanField(default=False, db_index=True)
    archived_at = models.DateTimeField(null=True, blank=True)
    archived_by = models.ForeignKey(User, null=True, blank=True, on_delete=models.SET_NULL, related_name='archived_%(class)ss')

    class Meta:
        abstract = True

    def archive(self, user=None):
        """Archive the instance."""
        self.archived = True
        self.archived_at = timezone.now()
        self.archived_by = user
        self.save()

    def unarchive(self):
        """Unarchive the instance."""
        self.archived = False
        self.archived_at = None
        self.archived_by = None
        self.save()


# =====================================================
# 🔹 SYSTEM & ANALYTICS MODELS
# =====================================================
class SiteConfig(models.Model):
    """Site-wide configuration settings."""
    name = models.CharField(max_length=100, default='SeovoSolutions')
    last_sync = models.DateTimeField(auto_now=True)
    system_status = models.CharField(
        max_length=20,
        choices=[('operational', 'Operational'), ('maintenance', 'Maintenance'), ('degraded', 'Degraded')],
        default='operational'
    )
    maintenance_mode = models.BooleanField(default=False)
    version = models.CharField(max_length=20, default='2.0.0')
    contact_email = models.EmailField(default='contact@seovosolutions.com')
    support_email = models.EmailField(default='support@seovosolutions.com')
    phone_number = models.CharField(max_length=20, blank=True, default='')
    address = models.TextField(blank=True, default='')
    social_links = models.JSONField(default=dict, blank=True)
    analytics_code = models.TextField(blank=True, default='')
    created_at = models.DateTimeField(default=timezone.now)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return f"{self.name} Configuration"

    class Meta:
        verbose_name = "Site Configuration"
        verbose_name_plural = "Site Configuration"


class SystemMetric(models.Model):
    """System metrics for monitoring and analytics."""
    METRIC_TYPES = [
        ('performance', 'Performance'),
        ('usage', 'Usage'),
        ('error', 'Error'),
        ('business', 'Business'),
        ('system', 'System'),
    ]
    
    metric_type = models.CharField(max_length=50, choices=METRIC_TYPES, default='usage', db_index=True)
    value = models.FloatField(default=0.0)
    metadata = models.JSONField(default=dict, blank=True)
    recorded_at = models.DateTimeField(default=timezone.now, db_index=True)

    def __str__(self):
        return f"{self.metric_type}: {self.value}"

    class Meta:
        ordering = ['-recorded_at']
        verbose_name = "System Metric"
        verbose_name_plural = "System Metrics"
        indexes = [
            models.Index(fields=['metric_type', 'recorded_at']),
        ]


class PageView(models.Model):
    """Tracks page views for analytics."""
    page_url = models.URLField(max_length=500)
    page_title = models.CharField(max_length=200, blank=True, default='')
    ip_address = models.GenericIPAddressField(null=True, blank=True)
    user_agent = models.TextField(blank=True, default='')
    referrer = models.URLField(blank=True, default='')
    session_id = models.CharField(max_length=100, blank=True, default='')
    user = models.ForeignKey(User, null=True, blank=True, on_delete=models.SET_NULL)
    duration = models.PositiveIntegerField(default=0, help_text="Time spent on page in seconds")
    created_at = models.DateTimeField(default=timezone.now, db_index=True)

    def __str__(self):
        return f"{self.page_url} - {self.created_at.strftime('%Y-%m-%d %H:%M')}"

    class Meta:
        ordering = ['-created_at']
        verbose_name = "Page View"
        verbose_name_plural = "Page Views"
        indexes = [
            models.Index(fields=['page_url', 'created_at']),
            models.Index(fields=['user', 'created_at']),
        ]


# =====================================================
# 🔹 CONTACTS / COMMUNICATION (Enhanced)
# =====================================================
class Contact(TimeStampedModel, ArchivableModel):
    """Stores inquiries or messages sent through the website."""
    CONTACT_TYPES = [
        ('general', 'General Inquiry'),
        ('support', 'Technical Support'),
        ('partnership', 'Partnership'),
        ('career', 'Career Opportunity'),
        ('training', 'Training Inquiry'),
        ('other', 'Other'),
    ]
    
    name = models.CharField(max_length=150, default='Anonymous')
    email = models.EmailField(default='no-email@example.com', db_index=True)
    phone = models.CharField(max_length=20, blank=True, default='')
    company = models.CharField(max_length=150, blank=True, default='')
    contact_type = models.CharField(max_length=50, choices=CONTACT_TYPES, default='general', db_index=True)
    subject = models.CharField(max_length=200, blank=True, default='No Subject')
    message = models.TextField(default='No message provided')
    responded = models.BooleanField(default=False, db_index=True)
    responded_at = models.DateTimeField(null=True, blank=True)
    responded_by = models.ForeignKey(User, null=True, blank=True, on_delete=models.SET_NULL, related_name='responded_contacts')
    response_notes = models.TextField(blank=True, default='')
    priority = models.CharField(
        max_length=20,
        choices=[('low', 'Low'), ('medium', 'Medium'), ('high', 'High'), ('urgent', 'Urgent')],
        default='medium',
        db_index=True
    )
    
    # Additional fields for better contact management
    interest_area = models.CharField(max_length=100, blank=True, default='')
    budget = models.CharField(max_length=100, blank=True, default='')
    timeframe = models.CharField(max_length=100, blank=True, default='')
    source = models.CharField(max_length=100, blank=True, default='website', db_index=True)
    newsletter_optin = models.BooleanField(default=False)
    ip_address = models.GenericIPAddressField(null=True, blank=True)
    user_agent = models.TextField(blank=True, default='')

    def mark_responded(self, user=None, notes=""):
        """Mark contact as responded."""
        self.responded = True
        self.responded_at = timezone.now()
        self.responded_by = user
        self.response_notes = notes
        self.save()

    def get_absolute_url(self):
        """Get admin URL for this contact."""
        return reverse('admin:website_contact_change', args=[self.id])

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'name': self.name,
            'email': self.email,
            'phone': self.phone,
            'company': self.company,
            'contact_type': self.contact_type,
            'subject': self.subject,
            'message': self.message,
            'responded': self.responded,
            'responded_at': self.responded_at.isoformat() if self.responded_at else None,
            'priority': self.priority,
            'interest_area': self.interest_area,
            'budget': self.budget,
            'timeframe': self.timeframe,
            'source': self.source,
            'newsletter_optin': self.newsletter_optin,
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
        }

    def __str__(self):
        return f"{self.name} <{self.email}> - {self.subject}"

    class Meta:
        ordering = ['-created_at']
        verbose_name = "Contact Submission"
        verbose_name_plural = "Contact Submissions"
        indexes = [
            models.Index(fields=['email', 'created_at']),
            models.Index(fields=['contact_type', 'responded']),
        ]


# =====================================================
# 🔹 TRAINING PROGRAMS - ENHANCED
# =====================================================
class TrainingProgram(TimeStampedModel, SoftDeletableModel, MetaDataModel):
    """Defines a training or educational program."""
    PROGRAM_TYPES = [
        ('workshop', 'Workshop'),
        ('certification', 'Certification'),
        ('webinar', 'Webinar'),
        ('bootcamp', 'Bootcamp'),
        ('course', 'Course'),
    ]
    
    PROGRAM_LEVELS = [
        ('beginner', 'Beginner'),
        ('intermediate', 'Intermediate'),
        ('advanced', 'Advanced'),
        ('all', 'All Levels'),
    ]

    PROGRAM_STATUS = [
        ('draft', 'Draft'),
        ('published', 'Published'),
        ('cancelled', 'Cancelled'),
        ('completed', 'Completed'),
    ]

    title = models.CharField(max_length=200, default='Untitled Training Program')
    description = models.TextField(default='Description not available')
    short_description = models.TextField(max_length=300, blank=True, default='')
    detailed_description = models.TextField(blank=True, default='')
    program_type = models.CharField(max_length=50, choices=PROGRAM_TYPES, default='course', db_index=True)
    level = models.CharField(max_length=50, choices=PROGRAM_LEVELS, default='all', db_index=True)
    status = models.CharField(max_length=20, choices=PROGRAM_STATUS, default='draft', db_index=True)
    start_date = models.DateField(null=True, blank=True, db_index=True)
    end_date = models.DateField(null=True, blank=True, db_index=True)
    duration = models.CharField(max_length=100, blank=True, default='', help_text="e.g., 6 weeks, 2 days")
    duration_weeks = models.PositiveIntegerField(default=12, help_text="Duration in weeks")
    hours_per_week = models.PositiveIntegerField(default=10, help_text="Hours per week")
    capacity = models.PositiveIntegerField(default=50)
    current_enrollment = models.PositiveIntegerField(default=0)
    instructor = models.CharField(max_length=150, blank=True, default='TBA')
    instructor_bio = models.TextField(blank=True, default='')
    price = models.DecimalField(max_digits=10, decimal_places=2, default=0.00, db_index=True)
    discount_price = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True)
    currency = models.CharField(max_length=10, default='USD')
    registration_open = models.BooleanField(default=True, db_index=True)
    registration_deadline = models.DateField(null=True, blank=True, db_index=True)
    featured = models.BooleanField(default=False, db_index=True)
    slug = models.SlugField(unique=True, blank=True, max_length=300)
    image = models.ImageField(upload_to='training/programs/%Y/%m/', null=True, blank=True)
    cover_image = models.ImageField(upload_to='training/covers/%Y/%m/', null=True, blank=True)
    learning_objectives = models.JSONField(default=list, blank=True)
    objectives = models.TextField(blank=True, default='')
    prerequisites = models.TextField(blank=True, default='')
    curriculum = models.JSONField(default=list, blank=True)
    certification = models.TextField(blank=True, default='')
    career_support = models.TextField(blank=True, default='')
    language = models.CharField(max_length=10, default='en', db_index=True)
    difficulty_score = models.PositiveSmallIntegerField(
        default=3,
        validators=[MinValueValidator(1), MaxValueValidator(5)],
        help_text="Difficulty level from 1 (easiest) to 5 (hardest)"
    )

    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.title)
            slug = base_slug
            counter = 1
            while TrainingProgram.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        
        # Auto-set status based on dates
        if self.start_date and self.end_date:
            today = timezone.now().date()
            if today > self.end_date:
                self.status = 'completed'
            elif today >= self.start_date:
                self.status = 'published'
        
        super().save(*args, **kwargs)

    def is_ongoing(self):
        """Check if the program is currently ongoing."""
        if self.start_date and self.end_date:
            today = timezone.now().date()
            return self.start_date <= today <= self.end_date
        return False

    def is_upcoming(self):
        """Check if the program is upcoming."""
        if self.start_date:
            return self.start_date > timezone.now().date()
        return False

    def is_registration_open(self):
        """Check if registration is currently open."""
        if not self.registration_open or self.status != 'published':
            return False
        if self.registration_deadline:
            return timezone.now().date() <= self.registration_deadline
        return True

    def available_spots(self):
        """Calculate available spots."""
        return max(0, self.capacity - self.current_enrollment)

    def enrollment_percentage(self):
        """Calculate enrollment percentage."""
        if self.capacity == 0:
            return 0
        return min(100, int((self.current_enrollment / self.capacity) * 100))

    def get_absolute_url(self):
        """Get frontend URL for this program."""
        return f"{settings.FRONTEND_URL}/training/{self.slug}"

    def image_url(self):
        """Return full image URL for frontend."""
        if self.image:
            return f"{settings.MEDIA_URL}{self.image.url}" if settings.DEBUG else f"{settings.FRONTEND_URL}{self.image.url}"
        return ""

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'title': self.title,
            'description': self.description,
            'short_description': self.short_description,
            'program_type': self.program_type,
            'level': self.level,
            'status': self.status,
            'start_date': self.start_date.isoformat() if self.start_date else None,
            'end_date': self.end_date.isoformat() if self.end_date else None,
            'duration': self.duration,
            'duration_weeks': self.duration_weeks,
            'hours_per_week': self.hours_per_week,
            'capacity': self.capacity,
            'current_enrollment': self.current_enrollment,
            'instructor': self.instructor,
            'price': float(self.price) if self.price else 0.0,
            'discount_price': float(self.discount_price) if self.discount_price else None,
            'currency': self.currency,
            'registration_open': self.registration_open,
            'registration_deadline': self.registration_deadline.isoformat() if self.registration_deadline else None,
            'featured': self.featured,
            'slug': self.slug,
            'image_url': self.image_url(),
            'learning_objectives': self.learning_objectives,
            'prerequisites': self.prerequisites,
            'curriculum': self.curriculum,
            'certification': self.certification,
            'career_support': self.career_support,
            'difficulty_score': self.difficulty_score,
            'is_ongoing': self.is_ongoing(),
            'is_upcoming': self.is_upcoming(),
            'is_registration_open': self.is_registration_open(),
            'available_spots': self.available_spots(),
            'enrollment_percentage': self.enrollment_percentage(),
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
        }

    def clean(self):
        """Validate model data."""
        if self.start_date and self.end_date and self.start_date > self.end_date:
            raise ValidationError("End date cannot be before start date.")
        if self.discount_price and self.discount_price >= self.price:
            raise ValidationError("Discount price must be less than regular price.")

    def __str__(self):
        return self.title

    class Meta:
        ordering = ['-created_at']
        verbose_name = "Training Program"
        verbose_name_plural = "Training Programs"
        indexes = [
            models.Index(fields=['program_type', 'status']),
            models.Index(fields=['level', 'featured']),
            models.Index(fields=['start_date', 'end_date']),
        ]


class TrainingRegistration(TimeStampedModel):
    """Stores individual user registrations for training programs."""
    STATUS_CHOICES = [
        ('pending', 'Pending'),
        ('confirmed', 'Confirmed'),
        ('waitlisted', 'Waitlisted'),
        ('cancelled', 'Cancelled'),
        ('completed', 'Completed'),
    ]

    PAYMENT_STATUS_CHOICES = [
        ('pending', 'Pending'),
        ('paid', 'Paid'),
        ('partial', 'Partial Payment'),
        ('refunded', 'Refunded'),
        ('cancelled', 'Cancelled'),
    ]

    full_name = models.CharField(max_length=150, default='Anonymous User')
    email = models.EmailField(default='no-email@example.com', db_index=True)
    phone = models.CharField(max_length=20, blank=True, default='')
    company = models.CharField(max_length=150, blank=True, default='')
    program = models.ForeignKey(TrainingProgram, on_delete=models.CASCADE, related_name="registrations", db_index=True)
    
    # Enhanced fields for better registration management
    experience_level = models.CharField(
        max_length=20,
        choices=[('beginner', 'Beginner'), ('intermediate', 'Intermediate'), ('advanced', 'Advanced')],
        blank=True, default='beginner'
    )
    learning_goals = models.TextField(blank=True, default='')
    preferred_schedule = models.CharField(
        max_length=20,
        choices=[
            ('weekdays', 'Weekdays'),
            ('weekends', 'Weekends'),
            ('evenings', 'Evenings'),
            ('flexible', 'Flexible'),
            ('intensive', 'Intensive Bootcamp')
        ],
        blank=True, default='flexible'
    )
    time_commitment = models.CharField(max_length=50, blank=True, default='10-15 hours per week')
    background = models.TextField(blank=True, default='')
    questions = models.TextField(blank=True, default='')
    payment_method = models.CharField(max_length=50, blank=True, default='credit_card')
    
    status = models.CharField(max_length=50, choices=STATUS_CHOICES, default='pending', db_index=True)
    payment_status = models.CharField(max_length=50, choices=PAYMENT_STATUS_CHOICES, default='pending', db_index=True)
    amount_paid = models.DecimalField(max_digits=10, decimal_places=2, default=0.00)
    notes = models.TextField(blank=True, default='')
    admin_notes = models.TextField(blank=True, default='')
    special_requirements = models.TextField(blank=True, default='')
    referral_source = models.CharField(max_length=200, blank=True, default='')
    contacted = models.BooleanField(default=False)
    contacted_at = models.DateTimeField(null=True, blank=True)
    
    # Additional metadata for tracking
    metadata = models.JSONField(default=dict, blank=True)
    source = models.CharField(max_length=100, default='website', db_index=True)
    submission_date = models.DateTimeField(default=timezone.now)
    registration_code = models.CharField(max_length=20, unique=True, blank=True, null=True)

    def save(self, *args, **kwargs):
        """Update program enrollment count when registration status changes."""
        # Generate unique registration code
        if not self.registration_code:
            self.registration_code = self._generate_registration_code()
        
        # Handle enrollment count changes
        if self.pk:
            try:
                old_instance = TrainingRegistration.objects.get(pk=self.pk)
                old_status = old_instance.status
                new_status = self.status
                
                # If status changed to/from confirmed, update enrollment count
                if old_status != new_status:
                    if old_status == 'confirmed':
                        self.program.current_enrollment = models.F('current_enrollment') - 1
                        self.program.save(update_fields=['current_enrollment'])
                    if new_status == 'confirmed':
                        self.program.current_enrollment = models.F('current_enrollment') + 1
                        self.program.save(update_fields=['current_enrollment'])
            except TrainingRegistration.DoesNotExist:
                # New instance
                if self.status == 'confirmed':
                    self.program.current_enrollment = models.F('current_enrollment') + 1
                    self.program.save(update_fields=['current_enrollment'])
        
        super().save(*args, **kwargs)

    def _generate_registration_code(self):
        """Generate unique registration code."""
        prefix = "REG"
        timestamp = timezone.now().strftime("%y%m%d")
        random_str = ''.join(random.choices(string.ascii_uppercase + string.digits, k=6))
        code = f"{prefix}{timestamp}{random_str}"
        
        # Ensure uniqueness
        while TrainingRegistration.objects.filter(registration_code=code).exists():
            random_str = ''.join(random.choices(string.ascii_uppercase + string.digits, k=6))
            code = f"{prefix}{timestamp}{random_str}"
        
        return code

    def delete(self, *args, **kwargs):
        """Handle enrollment count when registration is deleted."""
        if self.status == 'confirmed':
            self.program.current_enrollment = models.F('current_enrollment') - 1
            self.program.save(update_fields=['current_enrollment'])
        super().delete(*args, **kwargs)

    def get_absolute_url(self):
        """Get admin URL for this registration."""
        return reverse('admin:website_trainingregistration_change', args=[self.id])

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'full_name': self.full_name,
            'email': self.email,
            'phone': self.phone,
            'company': self.company,
            'program_id': self.program_id,
            'program_title': self.program.title if self.program else None,
            'experience_level': self.experience_level,
            'learning_goals': self.learning_goals,
            'preferred_schedule': self.preferred_schedule,
            'time_commitment': self.time_commitment,
            'background': self.background,
            'questions': self.questions,
            'payment_method': self.payment_method,
            'status': self.status,
            'payment_status': self.payment_status,
            'amount_paid': float(self.amount_paid),
            'notes': self.notes,
            'special_requirements': self.special_requirements,
            'referral_source': self.referral_source,
            'contacted': self.contacted,
            'contacted_at': self.contacted_at.isoformat() if self.contacted_at else None,
            'registration_code': self.registration_code,
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
        }

    def __str__(self):
        return f"{self.full_name} - {self.program.title}"

    class Meta:
        ordering = ['-created_at']
        verbose_name = "Training Registration"
        verbose_name_plural = "Training Registrations"
        indexes = [
            models.Index(fields=['email', 'status']),
            models.Index(fields=['program', 'status']),
            models.Index(fields=['registration_code']),
        ]


# =====================================================
# 🔹 POSTS / ARTICLES / BLOGS - ENHANCED
# =====================================================
class Category(TimeStampedModel):
    """Categories for blog posts and content with hierarchy support."""
    name = models.CharField(max_length=100, default='Uncategorized')
    slug = models.SlugField(unique=True, max_length=200)
    description = models.TextField(blank=True, default='')
    color = models.CharField(max_length=7, default='#3B82F6', help_text="Hex color code")
    is_active = models.BooleanField(default=True, db_index=True)
    parent = models.ForeignKey('self', null=True, blank=True, on_delete=models.CASCADE, related_name='children')
    order = models.PositiveIntegerField(default=0, help_text="Display order")
    post_count = models.PositiveIntegerField(default=0, editable=False)

    def save(self, *args, **kwargs):
        if not self.slug:
            self.slug = slugify(self.name)
        super().save(*args, **kwargs)

    def update_post_count(self):
        """Update post count for this category."""
        from django.db.models import Count
        count = Post.objects.filter(categories=self, published=True).count()
        Category.objects.filter(pk=self.pk).update(post_count=count)

    def get_absolute_url(self):
        """Get frontend URL for this category."""
        return f"{settings.FRONTEND_URL}/blog/category/{self.slug}"

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'name': self.name,
            'slug': self.slug,
            'description': self.description,
            'color': self.color,
            'is_active': self.is_active,
            'parent_id': self.parent_id,
            'order': self.order,
            'post_count': self.post_count,
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
        }

    def __str__(self):
        return self.name

    class Meta:
        ordering = ['order', 'name']
        verbose_name_plural = "Categories"
        indexes = [
            models.Index(fields=['is_active', 'parent']),
        ]


class Post(TimeStampedModel, SoftDeletableModel, MetaDataModel):
    """Stores blog posts, announcements, and AI-generated insights."""
    POST_TYPES = [
        ('blog', 'Blog Post'),
        ('news', 'News'),
        ('announcement', 'Announcement'),
        ('tutorial', 'Tutorial'),
        ('case_study', 'Case Study'),
        ('guide', 'Guide'),
    ]

    POST_STATUS = [
        ('draft', 'Draft'),
        ('published', 'Published'),
        ('scheduled', 'Scheduled'),
        ('archived', 'Archived'),
    ]

    title = models.CharField(max_length=250, default='Untitled Post')
    slug = models.SlugField(unique=True, blank=True, max_length=300)
    excerpt = models.TextField(max_length=500, blank=True, default='')
    content = models.TextField(default='Content not available')
    author = models.ForeignKey(User, null=True, blank=True, on_delete=models.SET_NULL, related_name='posts')
    categories = models.ManyToManyField(Category, blank=True, related_name='posts')
    post_type = models.CharField(max_length=50, choices=POST_TYPES, default='blog', db_index=True)
    status = models.CharField(max_length=20, choices=POST_STATUS, default='draft', db_index=True)
    published = models.BooleanField(default=False, db_index=True)
    published_at = models.DateTimeField(null=True, blank=True, db_index=True)
    scheduled_at = models.DateTimeField(null=True, blank=True, help_text="Schedule publication for future")
    featured = models.BooleanField(default=False, db_index=True)
    pinned = models.BooleanField(default=False, db_index=True)
    reading_time = models.PositiveIntegerField(default=5, help_text="Estimated reading time in minutes")
    image = models.ImageField(upload_to='posts/%Y/%m/', null=True, blank=True)
    cover_image = models.ImageField(upload_to='posts/covers/%Y/%m/', null=True, blank=True)
    attachments = models.ManyToManyField('FileUpload', blank=True, related_name='posts')
    
    # Additional metrics
    comment_count = models.PositiveIntegerField(default=0)
    like_count = models.PositiveIntegerField(default=0)
    share_count = models.PositiveIntegerField(default=0)
    language = models.CharField(max_length=10, default='en', db_index=True)
    word_count = models.PositiveIntegerField(default=0, editable=False)

    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.title)
            slug = base_slug
            counter = 1
            while Post.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        
        # Calculate word count
        self.word_count = len(self.content.split())
        
        # Set published status and timestamp
        if self.status == 'published' and not self.published_at:
            self.published_at = timezone.now()
            self.published = True
        elif self.status != 'published':
            self.published = False
            
        # Handle scheduled posts
        if self.scheduled_at and self.scheduled_at <= timezone.now() and self.status == 'scheduled':
            self.status = 'published'
            self.published_at = timezone.now()
            self.published = True
            
        super().save(*args, **kwargs)

    def get_absolute_url(self):
        """Get frontend URL for this post."""
        return f"{settings.FRONTEND_URL}/blog/{self.slug}"

    def image_url(self):
        """Return full image URL for frontend."""
        if self.image:
            return f"{settings.MEDIA_URL}{self.image.url}" if settings.DEBUG else f"{settings.FRONTEND_URL}{self.image.url}"
        return ""

    def is_scheduled(self):
        """Check if post is scheduled for future publication."""
        return self.status == 'scheduled' and self.scheduled_at and self.scheduled_at > timezone.now()

    def increment_share_count(self):
        """Atomically increment share count."""
        self.share_count = models.F('share_count') + 1
        self.save(update_fields=['share_count'])

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'title': self.title,
            'slug': self.slug,
            'excerpt': self.excerpt,
            'content': self.content,
            'author': self.author.username if self.author else 'Anonymous',
            'author_id': self.author_id,
            'categories': [cat.to_frontend_dict() for cat in self.categories.all()],
            'post_type': self.post_type,
            'status': self.status,
            'published': self.published,
            'published_at': self.published_at.isoformat() if self.published_at else None,
            'scheduled_at': self.scheduled_at.isoformat() if self.scheduled_at else None,
            'featured': self.featured,
            'pinned': self.pinned,
            'reading_time': self.reading_time,
            'image_url': self.image_url(),
            'comment_count': self.comment_count,
            'like_count': self.like_count,
            'share_count': self.share_count,
            'word_count': self.word_count,
            'is_scheduled': self.is_scheduled(),
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
            'meta_title': self.meta_title,
            'meta_description': self.meta_description,
            'tags': self.tags,
            'view_count': self.view_count,
        }

    def __str__(self):
        return self.title

    class Meta:
        ordering = ['-published_at', '-created_at']
        verbose_name = "Blog Post"
        verbose_name_plural = "Blog Posts"
        indexes = [
            models.Index(fields=['status', 'published_at']),
            models.Index(fields=['post_type', 'featured']),
            models.Index(fields=['author', 'created_at']),
        ]


# =====================================================
# 🔹 EVENTS - ENHANCED
# =====================================================
class Event(TimeStampedModel, MetaDataModel):
    """Event management — workshops, webinars, conferences."""
    EVENT_TYPES = [
        ('workshop', 'Workshop'),
        ('webinar', 'Webinar'),
        ('conference', 'Conference'),
        ('seminar', 'Seminar'),
        ('networking', 'Networking Event'),
        ('training', 'Training Session'),
        ('meetup', 'Community Meetup'),
    ]

    EVENT_STATUS = [
        ('draft', 'Draft'),
        ('scheduled', 'Scheduled'),
        ('ongoing', 'Ongoing'),
        ('completed', 'Completed'),
        ('cancelled', 'Cancelled'),
    ]

    title = models.CharField(max_length=200, default='Untitled Event')
    description = models.TextField(default='Event description not available')
    short_description = models.TextField(max_length=300, blank=True, default='')
    detailed_description = models.TextField(blank=True, default='')
    event_type = models.CharField(max_length=50, choices=EVENT_TYPES, default='workshop', db_index=True)
    status = models.CharField(max_length=20, choices=EVENT_STATUS, default='scheduled', db_index=True)
    start_time = models.DateTimeField(default=timezone.now, db_index=True)
    end_time = models.DateTimeField(null=True, blank=True, db_index=True)
    timezone = models.CharField(max_length=50, default='UTC')
    location = models.CharField(max_length=255, blank=True, default='TBA')
    virtual_link = models.URLField(blank=True, default='', help_text="Link for virtual events")
    is_virtual = models.BooleanField(default=False, db_index=True)
    is_hybrid = models.BooleanField(default=False)
    image = models.ImageField(upload_to="events/%Y/%m/", null=True, blank=True)
    organizer = models.ForeignKey(User, null=True, blank=True, on_delete=models.SET_NULL, related_name='organized_events')
    capacity = models.PositiveIntegerField(default=100)
    current_registrations = models.PositiveIntegerField(default=0)
    price = models.DecimalField(max_digits=10, decimal_places=2, default=0.00, db_index=True)
    currency = models.CharField(max_length=10, default='USD')
    registration_required = models.BooleanField(default=True)
    registration_deadline = models.DateTimeField(null=True, blank=True, db_index=True)
    registration_open = models.BooleanField(default=True, db_index=True)
    featured = models.BooleanField(default=False, db_index=True)
    published = models.BooleanField(default=True, db_index=True)

    def save(self, *args, **kwargs):
        # Set end_time to start_time + 2 hours if not provided
        if not self.end_time and self.start_time:
            self.end_time = self.start_time + timezone.timedelta(hours=2)
        
        # Auto-update status based on dates
        now = timezone.now()
        if self.start_time and self.end_time:
            if now < self.start_time:
                self.status = 'scheduled'
            elif self.start_time <= now <= self.end_time:
                self.status = 'ongoing'
            elif now > self.end_time:
                self.status = 'completed'
        
        super().save(*args, **kwargs)

    def is_upcoming(self):
        """Check if event is upcoming."""
        return self.start_time > timezone.now()

    def is_ongoing(self):
        """Check if event is currently ongoing."""
        now = timezone.now()
        return self.start_time <= now <= (self.end_time or self.start_time + timezone.timedelta(hours=2))

    def is_past(self):
        """Check if event has passed."""
        end_time = self.end_time or self.start_time + timezone.timedelta(hours=2)
        return end_time < timezone.now()

    def available_spots(self):
        """Calculate available spots."""
        return max(0, self.capacity - self.current_registrations)

    def registration_percentage(self):
        """Calculate registration percentage."""
        if self.capacity == 0:
            return 0
        return min(100, int((self.current_registrations / self.capacity) * 100))

    def get_absolute_url(self):
        """Get frontend URL for this event."""
        return f"{settings.FRONTEND_URL}/events/{self.id}"

    def image_url(self):
        """Full image URL for frontend display."""
        if self.image:
            return f"{settings.MEDIA_URL}{self.image.url}" if settings.DEBUG else f"{settings.FRONTEND_URL}{self.image.url}"
        return ""

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'title': self.title,
            'description': self.description,
            'short_description': self.short_description,
            'event_type': self.event_type,
            'status': self.status,
            'start_time': self.start_time.isoformat(),
            'end_time': self.end_time.isoformat() if self.end_time else None,
            'timezone': self.timezone,
            'location': self.location,
            'virtual_link': self.virtual_link,
            'is_virtual': self.is_virtual,
            'is_hybrid': self.is_hybrid,
            'image_url': self.image_url(),
            'organizer': self.organizer.username if self.organizer else 'SeovoSolutions',
            'capacity': self.capacity,
            'current_registrations': self.current_registrations,
            'price': float(self.price),
            'currency': self.currency,
            'registration_required': self.registration_required,
            'registration_deadline': self.registration_deadline.isoformat() if self.registration_deadline else None,
            'registration_open': self.registration_open,
            'featured': self.featured,
            'published': self.published,
            'is_upcoming': self.is_upcoming(),
            'is_ongoing': self.is_ongoing(),
            'is_past': self.is_past(),
            'available_spots': self.available_spots(),
            'registration_percentage': self.registration_percentage(),
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
            'meta_title': self.meta_title,
            'meta_description': self.meta_description,
            'tags': self.tags,
            'view_count': self.view_count,
        }

    def clean(self):
        """Validate event data."""
        if self.start_time and self.end_time and self.start_time >= self.end_time:
            raise ValidationError("End time must be after start time.")
        if self.registration_deadline and self.registration_deadline > self.start_time:
            raise ValidationError("Registration deadline cannot be after event start time.")

    def __str__(self):
        return self.title

    class Meta:
        ordering = ['-start_time']
        verbose_name = "Event"
        verbose_name_plural = "Events"
        indexes = [
            models.Index(fields=['event_type', 'status']),
            models.Index(fields=['start_time', 'end_time']),
            models.Index(fields=['is_virtual', 'published']),
        ]


# =====================================================
# 🔹 BLOVION POSTS (Frontend Media + Social Sync) - ENHANCED
# =====================================================
class BlovionPost(TimeStampedModel, MetaDataModel):
    """Posts for Blovion page — supports images, video, and social publishing."""
    POST_TYPES = [
        ('data_insight', 'Data Insight'),
        ('tutorial', 'Tutorial'),
        ('case_study', 'Case Study'),
        ('news', 'News'),
        ('announcement', 'Announcement'),
        ('thought_leadership', 'Thought Leadership'),
        ('analysis', 'Analysis'),
        ('trend', 'Trend Report'),
    ]

    POST_STATUS = [
        ('draft', 'Draft'),
        ('published', 'Published'),
        ('scheduled', 'Scheduled'),
        ('archived', 'Archived'),
    ]

    title = models.CharField(max_length=255, default='Untitled Blovion Post')
    caption = models.TextField(blank=True, default='')
    content = models.TextField(blank=True, default='')
    slug = models.SlugField(unique=True, blank=True, max_length=300)
    images = models.JSONField(default=list, blank=True)  # list of URLs
    video_url = models.URLField(blank=True, null=True)
    video_thumbnail = models.ImageField(upload_to='blovion/thumbnails/%Y/%m/', null=True, blank=True)
    post_type = models.CharField(max_length=50, choices=POST_TYPES, default='data_insight', db_index=True)
    status = models.CharField(max_length=20, choices=POST_STATUS, default='draft', db_index=True)
    category = models.CharField(max_length=100, blank=True, default='', db_index=True)
    categories = models.ManyToManyField(Category, blank=True, related_name='blovion_posts')
    published = models.BooleanField(default=False, db_index=True)
    published_at = models.DateTimeField(default=timezone.now, db_index=True)
    scheduled_at = models.DateTimeField(null=True, blank=True, help_text="Schedule publication for future")
    featured = models.BooleanField(default=False, db_index=True)
    social_posted = models.BooleanField(default=False)
    social_platforms = models.JSONField(default=list, blank=True, help_text="Platforms where post was shared")
    social_links = models.JSONField(default=dict, blank=True)
    read_time = models.PositiveIntegerField(default=5, help_text="Estimated reading time in minutes")
    author = models.CharField(max_length=100, default='SeovoSolutions Team')
    tools = models.JSONField(default=list, blank=True, help_text="Tools/technologies used")
    dataset_size = models.CharField(max_length=100, blank=True, default='')
    methodology = models.CharField(max_length=200, blank=True, default='')
    complexity = models.CharField(
        max_length=20,
        choices=[('beginner', 'Beginner'), ('intermediate', 'Intermediate'), ('advanced', 'Advanced')],
        default='intermediate',
        db_index=True
    )
    metadata = models.JSONField(default=dict, blank=True)
    
    # Additional metrics
    like_count = models.PositiveIntegerField(default=0)
    share_count = models.PositiveIntegerField(default=0)
    comment_count = models.PositiveIntegerField(default=0)
    word_count = models.PositiveIntegerField(default=0, editable=False)

    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.title)
            slug = base_slug
            counter = 1
            while BlovionPost.objects.filter(slug=slug).exclude(id=self.id).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        
        # Calculate word count
        self.word_count = len((self.content or '').split()) + len((self.caption or '').split())
        
        # Set published status and timestamp
        if self.status == 'published' and not self.published_at:
            self.published_at = timezone.now()
            self.published = True
        elif self.status != 'published':
            self.published = False
            
        # Handle scheduled posts
        if self.scheduled_at and self.scheduled_at <= timezone.now() and self.status == 'scheduled':
            self.status = 'published'
            self.published_at = timezone.now()
            self.published = True
            
        super().save(*args, **kwargs)

    def get_absolute_url(self):
        """Frontend URL for sharing."""
        return f"{settings.FRONTEND_URL}/blovion/{self.slug}"

    def image_list(self):
        """Ensure image URLs are absolute (for React frontend)."""
        if not self.images:
            return []
        base = getattr(settings, "FRONTEND_URL", "")
        return [url if url.startswith("http") else f"{base}{url}" for url in self.images]

    def image_count(self):
        """Get count of images for frontend."""
        return len(self.images) if self.images else 0

    def is_recent(self):
        """Check if post was published in the last 7 days."""
        if self.published_at:
            return (timezone.now() - self.published_at).days <= 7
        return False

    def increment_share_count(self):
        """Atomically increment share count."""
        self.share_count = models.F('share_count') + 1
        self.save(update_fields=['share_count'])

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'title': self.title,
            'caption': self.caption,
            'content': self.content,
            'slug': self.slug,
            'images': self.image_list(),
            'video_url': self.video_url,
            'post_type': self.post_type,
            'status': self.status,
            'category': self.category,
            'categories': [cat.to_frontend_dict() for cat in self.categories.all()],
            'published': self.published,
            'published_at': self.published_at.isoformat() if self.published_at else None,
            'scheduled_at': self.scheduled_at.isoformat() if self.scheduled_at else None,
            'featured': self.featured,
            'social_posted': self.social_posted,
            'social_platforms': self.social_platforms,
            'social_links': self.social_links,
            'read_time': self.read_time,
            'author': self.author,
            'tools': self.tools,
            'dataset_size': self.dataset_size,
            'methodology': self.methodology,
            'complexity': self.complexity,
            'like_count': self.like_count,
            'share_count': self.share_count,
            'comment_count': self.comment_count,
            'word_count': self.word_count,
            'image_count': self.image_count(),
            'is_recent': self.is_recent(),
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
            'meta_title': self.meta_title,
            'meta_description': self.meta_description,
            'tags': self.tags,
            'view_count': self.view_count,
        }

    def __str__(self):
        return self.title

    class Meta:
        ordering = ["-published_at", "-created_at"]
        verbose_name = "Blovion Post"
        verbose_name_plural = "Blovion Posts"
        indexes = [
            models.Index(fields=['post_type', 'status']),
            models.Index(fields=['complexity', 'featured']),
            models.Index(fields=['published_at', 'published']),
        ]


# =====================================================
# 🔹 TESTIMONIALS - ENHANCED
# =====================================================
class Testimonial(TimeStampedModel):
    """Stores feedback and reviews from clients or participants."""
    name = models.CharField(max_length=150, default='Anonymous')
    email = models.EmailField(blank=True, default='')
    company = models.CharField(max_length=150, blank=True, default='')
    position = models.CharField(max_length=150, blank=True, default='')
    message = models.TextField(default='No testimonial message')
    rating = models.PositiveSmallIntegerField(
        default=5, 
        validators=[MinValueValidator(1), MaxValueValidator(5)],
        db_index=True
    )
    approved = models.BooleanField(default=False, db_index=True)
    featured = models.BooleanField(default=False, db_index=True)
    project_type = models.CharField(max_length=100, blank=True, default='', help_text="e.g., SEO, Training, Consulting")
    image = models.ImageField(upload_to='testimonials/%Y/%m/', null=True, blank=True)
    website = models.URLField(blank=True, default='')
    location = models.CharField(max_length=100, blank=True, default='')
    display_order = models.PositiveIntegerField(default=0, help_text="Order in which testimonials are displayed")

    def image_url(self):
        """Return full image URL for frontend."""
        if self.image:
            return f"{settings.MEDIA_URL}{self.image.url}" if settings.DEBUG else f"{settings.FRONTEND_URL}{self.image.url}"
        return ""

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'name': self.name,
            'email': self.email,
            'company': self.company,
            'position': self.position,
            'message': self.message,
            'rating': self.rating,
            'approved': self.approved,
            'featured': self.featured,
            'project_type': self.project_type,
            'image_url': self.image_url(),
            'website': self.website,
            'location': self.location,
            'display_order': self.display_order,
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
        }

    def __str__(self):
        return f"{self.name} - {self.company or 'Independent'}"

    class Meta:
        ordering = ['display_order', '-created_at']
        verbose_name = "Testimonial"
        verbose_name_plural = "Testimonials"
        indexes = [
            models.Index(fields=['approved', 'featured']),
            models.Index(fields=['rating', 'created_at']),
        ]


# =====================================================
# 🔹 FILES & MEDIA - ENHANCED
# =====================================================
class FileUpload(TimeStampedModel):
    """Manages uploaded files and documents with improved file handling."""
    FILE_CATEGORIES = [
        ('document', 'Document'),
        ('image', 'Image'),
        ('video', 'Video'),
        ('audio', 'Audio'),
        ('archive', 'Archive'),
        ('other', 'Other'),
    ]

    file = models.FileField(upload_to="uploads/%Y/%m/%d/")
    uploaded_by = models.ForeignKey(User, null=True, blank=True, on_delete=models.SET_NULL, related_name='uploaded_files')
    title = models.CharField(max_length=255, blank=True, default='Untitled File')
    file_name = models.CharField(max_length=255, blank=True, default='')
    description = models.TextField(blank=True, default='')
    category = models.CharField(max_length=100, choices=FILE_CATEGORIES, default='document', db_index=True)
    size_kb = models.FloatField(default=0, db_index=True)
    file_size = models.PositiveBigIntegerField(default=0)
    file_type = models.CharField(max_length=50, blank=True, default='', db_index=True)
    is_public = models.BooleanField(default=False, db_index=True)
    metadata = models.JSONField(default=dict, blank=True)
    access_count = models.PositiveIntegerField(default=0)
    last_accessed = models.DateTimeField(null=True, blank=True)

    def save(self, *args, **kwargs):
        if self.file:
            if not self.size_kb:
                self.size_kb = round(self.file.size / 1024, 2)
                self.file_size = self.file.size
            
            if not self.file_type:
                name_parts = self.file.name.split('.')
                if len(name_parts) > 1:
                    self.file_type = name_parts[-1].lower()
            
            if not self.title:
                self.title = os.path.basename(self.file.name)
            
            if not self.file_name:
                self.file_name = os.path.basename(self.file.name)
                
        super().save(*args, **kwargs)

    def increment_access_count(self):
        """Atomically increment access count."""
        self.access_count = models.F('access_count') + 1
        self.last_accessed = timezone.now()
        self.save(update_fields=['access_count', 'last_accessed'])

    def file_url(self):
        """Return full URL for frontend."""
        if self.file:
            return f"{settings.MEDIA_URL}{self.file.url}" if settings.DEBUG else f"{settings.FRONTEND_URL}{self.file.url}"
        return ""

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'file_url': self.file_url(),
            'title': self.title,
            'file_name': self.file_name,
            'description': self.description,
            'category': self.category,
            'size_kb': self.size_kb,
            'file_size': self.file_size,
            'file_type': self.file_type,
            'is_public': self.is_public,
            'access_count': self.access_count,
            'last_accessed': self.last_accessed.isoformat() if self.last_accessed else None,
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
        }

    def __str__(self):
        return self.title or self.file.name

    class Meta:
        ordering = ['-created_at']
        verbose_name = "File Upload"
        verbose_name_plural = "File Uploads"
        indexes = [
            models.Index(fields=['category', 'file_type']),
            models.Index(fields=['uploaded_by', 'created_at']),
        ]


# =====================================================
# 🔹 AI / SEOVOSOLUTIONS ASSISTANT MODELS
# =====================================================
class ConversationLog(TimeStampedModel):
    """Stores AI conversation logs for analytics and improvement."""
    user = models.ForeignKey(User, null=True, blank=True, on_delete=models.SET_NULL, related_name='conversations')
    user_identifier = models.CharField(max_length=255, blank=True, default='', db_index=True)
    session_id = models.CharField(max_length=100, blank=True, default='', db_index=True)
    user_message = models.TextField()
    bot_reply = models.TextField()
    intent = models.CharField(max_length=100, blank=True, default='', db_index=True)
    confidence = models.FloatField(default=0.0)
    sources = models.JSONField(default=list, blank=True)
    metadata = models.JSONField(default=dict, blank=True)
    user_satisfaction = models.IntegerField(
        null=True, blank=True,
        validators=[MinValueValidator(1), MaxValueValidator(5)],
        help_text="User rating 1-5"
    )
    response_time = models.FloatField(default=0.0, help_text="Response time in seconds")

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'user_id': self.user_id,
            'user_identifier': self.user_identifier,
            'session_id': self.session_id,
            'user_message': self.user_message,
            'bot_reply': self.bot_reply,
            'intent': self.intent,
            'confidence': self.confidence,
            'sources': self.sources,
            'user_satisfaction': self.user_satisfaction,
            'response_time': self.response_time,
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
        }

    def __str__(self):
        return f"Conversation {self.id} - {self.user_identifier}"

    class Meta:
        ordering = ['-created_at']
        verbose_name = "Conversation Log"
        verbose_name_plural = "Conversation Logs"
        indexes = [
            models.Index(fields=['user_identifier', 'created_at']),
            models.Index(fields=['intent', 'confidence']),
        ]


class AssistantFeedback(TimeStampedModel):
    """Stores user feedback for AI assistant responses."""
    conversation = models.ForeignKey(
        ConversationLog, 
        on_delete=models.CASCADE, 
        related_name='feedbacks',
        null=True,  # ✅ Allow null for existing rows
        blank=True  # ✅ Allow blank in forms
    )
    helpful = models.BooleanField(default=True)
    rating = models.IntegerField(
        null=True, 
        blank=True,
        validators=[MinValueValidator(1), MaxValueValidator(5)]
    )
    comment = models.TextField(blank=True, default='')
    improvements_suggested = models.TextField(blank=True, default='')
    would_use_again = models.BooleanField(default=True)
    user_email = models.EmailField(blank=True, default='')

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'conversation_id': self.conversation_id,
            'helpful': self.helpful,
            'rating': self.rating,
            'comment': self.comment,
            'improvements_suggested': self.improvements_suggested,
            'would_use_again': self.would_use_again,
            'user_email': self.user_email,
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
        }

    def __str__(self):
        return f"Feedback for Conversation {self.conversation_id}"

    class Meta:
        ordering = ['-created_at']
        verbose_name = "Assistant Feedback"
        verbose_name_plural = "Assistant Feedbacks"
        indexes = [
            models.Index(fields=['conversation', 'helpful']),
            models.Index(fields=['rating', 'created_at']),
        ]


class Document(TimeStampedModel):
    """Stores documents for AI semantic search / RAG."""
    title = models.CharField(max_length=500, default='Untitled Document')
    text = models.TextField(default='')
    source = models.CharField(max_length=100, default='manual')
    source_id = models.CharField(max_length=100, blank=True, default='')
    chunk_index = models.IntegerField(default=0)
    total_chunks = models.IntegerField(default=1)
    embedding_vector = models.BinaryField(null=True, blank=True)
    embedding_dim = models.IntegerField(default=768)
    language = models.CharField(max_length=10, default='en')
    is_active = models.BooleanField(default=True, db_index=True)

    def __str__(self):
        return self.title

    class Meta:
        ordering = ['source', 'chunk_index']
        verbose_name = "Document"
        verbose_name_plural = "Documents"
        indexes = [
            models.Index(fields=['source', 'is_active']),
            models.Index(fields=['language', 'is_active']),
        ]


# =====================================================
# 🔹 NOTIFICATIONS
# =====================================================
class Notification(TimeStampedModel):
    """System notifications for users."""
    NOTIFICATION_TYPES = [
        ('info', 'Information'),
        ('success', 'Success'),
        ('warning', 'Warning'),
        ('error', 'Error'),
        ('system', 'System Update'),
    ]

    type = models.CharField(max_length=20, choices=NOTIFICATION_TYPES, default='info')
    title = models.CharField(max_length=200, default='Notification')
    message = models.TextField(default='')
    payload = models.JSONField(default=dict, blank=True)
    priority = models.CharField(
        max_length=20,
        choices=[('low', 'Low'), ('medium', 'Medium'), ('high', 'High'), ('urgent', 'Urgent')],
        default='medium'
    )
    read = models.BooleanField(default=False, db_index=True)
    read_at = models.DateTimeField(null=True, blank=True)
    recipient = models.ForeignKey(
        User, 
        on_delete=models.CASCADE, 
        related_name='notifications',
        null=True,  # ✅ ADDED: Allow null for existing rows
        blank=True  # ✅ ADDED: Allow blank in forms
    )
    action_url = models.URLField(blank=True, default='')
    expires_at = models.DateTimeField(null=True, blank=True)

    def mark_as_read(self):
        """Mark notification as read."""
        self.read = True
        self.read_at = timezone.now()
        self.save()

    def to_frontend_dict(self):
        """Convert to frontend-friendly dictionary."""
        return {
            'id': self.id,
            'type': self.type,
            'title': self.title,
            'message': self.message,
            'payload': self.payload,
            'priority': self.priority,
            'read': self.read,
            'read_at': self.read_at.isoformat() if self.read_at else None,
            'recipient_id': self.recipient_id,
            'action_url': self.action_url,
            'expires_at': self.expires_at.isoformat() if self.expires_at else None,
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat(),
        }

    def __str__(self):
        return f"{self.type}: {self.title}"

    class Meta:
        ordering = ['-created_at']
        verbose_name = "Notification"
        verbose_name_plural = "Notifications"
        indexes = [
            models.Index(fields=['recipient', 'read']),
            models.Index(fields=['type', 'priority']),
        ]
        
