# website/serializers.py
from rest_framework import serializers
from django.contrib.auth import get_user_model
from django.urls import reverse
from django.conf import settings
from .models import (
    BlovionPost, Contact, TrainingProgram, TrainingRegistration,
    FileUpload, Post, Event, Testimonial,
    Notification, Document, ConversationLog, AssistantFeedback,
    Category, PageView, SystemMetric, SiteConfig
)

User = get_user_model()

# ----------------------------------------------------------------------
# 🔹 ENHANCED BASE SERIALIZER WITH URL INTEGRATION
# ----------------------------------------------------------------------

class EnhancedModelSerializer(serializers.ModelSerializer):
    """Enhanced base serializer with URL integration and frontend compatibility."""
    
    def to_representation(self, instance):
        """Enhanced representation with URL integration."""
        data = super().to_representation(instance)
        
        # Add API URLs for frontend integration
        if hasattr(instance, 'get_absolute_url'):
            data['url'] = instance.get_absolute_url()
        
        # Add frontend-compatible data if available
        if hasattr(instance, 'to_frontend_dict'):
            frontend_data = instance.to_frontend_dict()
            data.update(frontend_data)
        
        return data

# ----------------------------------------------------------------------
# 🔹 CORE & SYSTEM MODELS - UPDATED FOR URL INTEGRATION
# ----------------------------------------------------------------------

class CategorySerializer(EnhancedModelSerializer):
    """Category for organizing content with URL integration."""
    api_url = serializers.SerializerMethodField()
    
    class Meta:
        model = Category
        fields = [
            "id", "name", "slug", "description", "color", "is_active",
            "created_at", "updated_at", "api_url"
        ]
        read_only_fields = ["id", "slug", "created_at", "updated_at", "api_url"]
    
    def get_api_url(self, obj):
        return reverse('category-detail', kwargs={'pk': obj.pk})


class SiteConfigSerializer(EnhancedModelSerializer):
    """Site configuration and settings with enhanced frontend integration."""
    class Meta:
        model = SiteConfig
        fields = [
            "id", "name", "last_sync", "system_status", "maintenance_mode",
            "version", "contact_email", "support_email", "phone_number",
            "address", "social_links", "analytics_code"
        ]
        read_only_fields = ["id", "last_sync"]


class SystemMetricSerializer(EnhancedModelSerializer):
    """System metrics and analytics data with enhanced representation."""
    class Meta:
        model = SystemMetric
        fields = [
            "id", "metric_type", "value", "metadata", "recorded_at"
        ]
        read_only_fields = ["id"]


class PageViewSerializer(EnhancedModelSerializer):
    """Page view tracking for analytics with enhanced data."""
    user_name = serializers.CharField(source="user.get_full_name", read_only=True)
    
    class Meta:
        model = PageView
        fields = [
            "id", "page_url", "page_title", "ip_address", "user_agent", 
            "referrer", "session_id", "user", "user_name", "duration", "created_at"
        ]
        read_only_fields = ["id", "created_at", "user_name"]


# ----------------------------------------------------------------------
# 🔹 CONTACTS & USER INTERACTIONS - UPDATED FOR API INTEGRATION
# ----------------------------------------------------------------------

class ContactSerializer(EnhancedModelSerializer):
    """Enhanced contact form submissions with API URL integration."""
    api_url = serializers.SerializerMethodField()
    status_display = serializers.CharField(source="get_priority_display", read_only=True)
    
    class Meta:
        model = Contact
        fields = [
            "id", "name", "email", "phone", "company", "contact_type", 
            "subject", "message", "responded", "responded_at", "response_notes",
            "priority", "status_display", "archived", "archived_at", 
            "created_at", "updated_at", "api_url"
        ]
        read_only_fields = [
            "id", "responded", "responded_at", "created_at", "updated_at", 
            "api_url", "status_display"
        ]

    def get_api_url(self, obj):
        return reverse('contacts-detail', kwargs={'pk': obj.pk})

    def validate_email(self, value):
        """Enhanced email validation."""
        if not value or "@" not in value:
            raise serializers.ValidationError("A valid email address is required.")
        
        # Basic email format validation
        if len(value) < 5 or "." not in value.split("@")[-1]:
            raise serializers.ValidationError("Please provide a valid email address.")
        
        return value.lower().strip()

    def validate_phone(self, value):
        """Enhanced phone number validation."""
        if value and len(value.strip()) < 5:
            raise serializers.ValidationError("Please provide a valid phone number.")
        return value

    def validate_message(self, value):
        """Enhanced message validation."""
        if len(value.strip()) < 10:
            raise serializers.ValidationError("Message is too short. Please provide more details.")
        
        if len(value.strip()) > 5000:
            raise serializers.ValidationError("Message is too long. Please keep it under 5000 characters.")
        
        return value.strip()


# ----------------------------------------------------------------------
# 🔹 TRAINING PROGRAMS / TRAIN-MENTOR PAGE - ENHANCED FOR URLS
# ----------------------------------------------------------------------

class TrainingProgramSerializer(EnhancedModelSerializer):
    """Enhanced training program serializer with URL integration."""
    instructor_name = serializers.CharField(source="instructor.get_full_name", read_only=True)
    current_enrollment_count = serializers.IntegerField(source="current_enrollment", read_only=True)
    is_registration_open = serializers.BooleanField(source="registration_open", read_only=True)
    available_spots = serializers.IntegerField(read_only=True)
    is_ongoing = serializers.BooleanField(read_only=True)
    is_upcoming = serializers.BooleanField(read_only=True)
    api_url = serializers.SerializerMethodField()
    registrations_url = serializers.SerializerMethodField()
    
    class Meta:
        model = TrainingProgram
        fields = [
            "id", "title", "slug", "description", "short_description",
            "instructor", "instructor_name", "program_type", "level",
            "duration", "capacity", "current_enrollment", "current_enrollment_count",
            "available_spots", "start_date", "end_date", "price", "discount_price",
            "registration_open", "is_registration_open", "featured", "image",
            "learning_objectives", "prerequisites", "curriculum", "is_ongoing",
            "is_upcoming", "meta_title", "meta_description", "tags", "view_count",
            "is_active", "created_at", "updated_at", "api_url", "registrations_url"
        ]
        read_only_fields = [
            "id", "slug", "current_enrollment", "view_count", 
            "created_at", "updated_at", "current_enrollment_count",
            "is_registration_open", "available_spots", "is_ongoing", "is_upcoming",
            "api_url", "registrations_url"
        ]

    def get_api_url(self, obj):
        return reverse('training-programs-detail', kwargs={'pk': obj.pk})

    def get_registrations_url(self, obj):
        return f"{reverse('training-list')}?program={obj.pk}"

    def validate_price(self, value):
        """Validate pricing information."""
        if value is not None and value < 0:
            raise serializers.ValidationError("Price cannot be negative.")
        return value

    def validate_capacity(self, value):
        """Validate capacity constraints."""
        if value <= 0:
            raise serializers.ValidationError("Capacity must be positive.")
        return value

    def validate_dates(self, data):
        """Validate program date logic."""
        if data.get('start_date') and data.get('end_date'):
            if data['start_date'] >= data['end_date']:
                raise serializers.ValidationError("End date must be after start date.")
        return data

    def create(self, validated_data):
        """Override create to include date validation."""
        self.validate_dates(validated_data)
        return super().create(validated_data)

    def update(self, instance, validated_data):
        """Override update to include date validation."""
        self.validate_dates(validated_data)
        return super().update(instance, validated_data)


class TrainingRegistrationSerializer(EnhancedModelSerializer):
    """Enhanced training registration with URL integration."""
    program_title = serializers.CharField(source="program.title", read_only=True)
    program_type = serializers.CharField(source="program.program_type", read_only=True)
    program_api_url = serializers.CharField(source="program.get_absolute_url", read_only=True)
    status_display = serializers.CharField(source="get_status_display", read_only=True)
    payment_status_display = serializers.CharField(source="get_payment_status_display", read_only=True)
    api_url = serializers.SerializerMethodField()
    
    class Meta:
        model = TrainingRegistration
        fields = [
            "id", "full_name", "email", "phone", "company", "program", "program_title",
            "program_type", "program_api_url", "status", "status_display", "payment_status", 
            "payment_status_display", "amount_paid", "notes", "special_requirements", 
            "referral_source", "contacted", "contacted_at", "created_at", "updated_at", "api_url"
        ]
        read_only_fields = [
            "id", "created_at", "updated_at", "program_title", "program_type",
            "status_display", "payment_status_display", "program_api_url", "api_url"
        ]

    def get_api_url(self, obj):
        return reverse('training-detail', kwargs={'pk': obj.pk})

    def validate_email(self, value):
        """Enhanced email validation."""
        if not value or "@" not in value:
            raise serializers.ValidationError("A valid email address is required.")
        
        # Check for existing registrations
        if self.context.get('request') and self.context['request'].method == 'POST':
            program_id = self.initial_data.get('program')
            if program_id:
                existing = TrainingRegistration.objects.filter(
                    email=value, 
                    program_id=program_id
                ).exists()
                if existing:
                    raise serializers.ValidationError(
                        "You have already registered for this program with this email."
                    )
        
        return value.lower().strip()

    def validate_phone(self, value):
        """Enhanced phone validation."""
        if value and len(value.strip()) < 5:
            raise serializers.ValidationError("Please provide a valid phone number.")
        return value


# ----------------------------------------------------------------------
# 🔹 FILES & MEDIA - ENHANCED FOR URL INTEGRATION
# ----------------------------------------------------------------------

class FileUploadSerializer(EnhancedModelSerializer):
    """Enhanced file upload serializer with URL integration."""
    uploaded_by_name = serializers.CharField(source="uploaded_by.get_full_name", read_only=True)
    file_size_mb = serializers.SerializerMethodField()
    file_url = serializers.SerializerMethodField()
    api_url = serializers.SerializerMethodField()
    
    class Meta:
        model = FileUpload
        fields = [
            "id", "file", "file_url", "title", "description", "category",
            "uploaded_by", "uploaded_by_name", "size_kb", "file_size_mb",
            "file_type", "is_public", "created_at", "updated_at", "api_url"
        ]
        read_only_fields = [
            "id", "uploaded_by", "size_kb", "file_type", 
            "created_at", "updated_at", "file_size_mb", "uploaded_by_name",
            "file_url", "api_url"
        ]

    def get_api_url(self, obj):
        return reverse('file-uploads-detail', kwargs={'pk': obj.pk})

    def get_file_url(self, obj):
        if obj.file:
            return obj.file.url
        return None

    def get_file_size_mb(self, obj):
        """Convert file size to MB."""
        if obj.size_kb:
            return round(obj.size_kb / 1024, 2)
        return 0

    def validate_file(self, value):
        """Enhanced file validation."""
        if value:
            # Size validation (25MB max)
            max_size = 25 * 1024 * 1024
            if value.size > max_size:
                raise serializers.ValidationError("File too large (max 25MB).")
        
        return value


# ----------------------------------------------------------------------
# 🔹 POSTS / BLOG / ANNOUNCEMENTS - ENHANCED FOR URL INTEGRATION
# ----------------------------------------------------------------------

class PostSerializer(EnhancedModelSerializer):
    """Enhanced blog post serializer with URL integration."""
    author_name = serializers.CharField(source="author.get_full_name", read_only=True)
    attachments = FileUploadSerializer(many=True, read_only=True)
    categories = CategorySerializer(many=True, read_only=True)
    category_ids = serializers.PrimaryKeyRelatedField(
        many=True, 
        queryset=Category.objects.all(),
        source='categories',
        write_only=True,
        required=False
    )
    api_url = serializers.SerializerMethodField()
    image_url = serializers.SerializerMethodField()
    
    class Meta:
        model = Post
        fields = [
            "id", "title", "slug", "content", "excerpt", "author", "author_name",
            "image", "image_url", "attachments", "categories", "category_ids", "post_type",
            "published", "published_at", "meta_title", "meta_description",
            "tags", "view_count", "reading_time", "featured", "is_active",
            "created_at", "updated_at", "api_url"
        ]
        read_only_fields = [
            "id", "slug", "author", "view_count", "created_at", "updated_at",
            "api_url", "image_url", "author_name"
        ]

    def get_api_url(self, obj):
        return reverse('posts-detail', kwargs={'pk': obj.pk})

    def get_image_url(self, obj):
        if obj.image:
            return obj.image.url
        return None

    def validate_content(self, value):
        """Content validation."""
        if len(value.strip()) < 100:
            raise serializers.ValidationError("Content is too short.")
        return value

    def create(self, validated_data):
        """Set author automatically."""
        if not validated_data.get('author'):
            request = self.context.get('request')
            if request and request.user.is_authenticated:
                validated_data['author'] = request.user
        return super().create(validated_data)


# ----------------------------------------------------------------------
# 🔹 EVENTS PAGE - ENHANCED FOR URL INTEGRATION
# ----------------------------------------------------------------------

class EventSerializer(EnhancedModelSerializer):
    """Enhanced event serializer with URL integration."""
    organizer_name = serializers.CharField(source="organizer.get_full_name", read_only=True)
    is_upcoming = serializers.BooleanField(read_only=True)
    is_ongoing = serializers.BooleanField(read_only=True)
    is_past = serializers.BooleanField(read_only=True)
    available_spots = serializers.IntegerField(read_only=True)
    api_url = serializers.SerializerMethodField()
    image_url = serializers.SerializerMethodField()
    registrations_url = serializers.SerializerMethodField()
    
    class Meta:
        model = Event
        fields = [
            "id", "title", "description", "short_description", "event_type",
            "start_time", "end_time", "location", "virtual_link",
            "image", "image_url", "is_virtual", "is_hybrid", "organizer", "organizer_name",
            "capacity", "current_registrations", "available_spots", "price",
            "registration_open", "featured", "published", "is_upcoming",
            "is_ongoing", "is_past", "meta_title", "meta_description", "tags", 
            "view_count", "created_at", "updated_at", "api_url", "registrations_url"
        ]
        read_only_fields = [
            "id", "organizer", "view_count", "current_registrations",
            "is_upcoming", "is_ongoing", "is_past", "available_spots",
            "created_at", "updated_at", "api_url", "image_url", "registrations_url",
            "organizer_name"
        ]

    def get_api_url(self, obj):
        return reverse('events-detail', kwargs={'pk': obj.pk})

    def get_image_url(self, obj):
        if obj.image:
            return obj.image.url
        return None

    def get_registrations_url(self, obj):
        return f"{reverse('training-list')}?event={obj.pk}"

    def validate_dates(self, data):
        """Validate event date logic."""
        if data.get('start_time') and data.get('end_time'):
            if data['start_time'] >= data['end_time']:
                raise serializers.ValidationError("End time must be after start time.")
        return data

    def create(self, validated_data):
        """Set organizer automatically and validate dates."""
        self.validate_dates(validated_data)
        
        if not validated_data.get('organizer'):
            request = self.context.get('request')
            if request and request.user.is_authenticated:
                validated_data['organizer'] = request.user
        
        return super().create(validated_data)

    def update(self, instance, validated_data):
        """Validate dates on update."""
        self.validate_dates(validated_data)
        return super().update(instance, validated_data)


# ----------------------------------------------------------------------
# 🔹 TESTIMONIALS PAGE - ENHANCED FOR URL INTEGRATION
# ----------------------------------------------------------------------

class TestimonialSerializer(EnhancedModelSerializer):
    """Enhanced testimonial serializer with URL integration."""
    api_url = serializers.SerializerMethodField()
    image_url = serializers.SerializerMethodField()
    
    class Meta:
        model = Testimonial
        fields = [
            "id", "name", "company", "position", "project_type",
            "message", "rating", "image", "image_url", "website", "approved", "featured",
            "created_at", "updated_at", "api_url"
        ]
        read_only_fields = [
            "id", "approved", "created_at", "updated_at", "api_url", "image_url"
        ]

    def get_api_url(self, obj):
        return reverse('testimonials-detail', kwargs={'pk': obj.pk})

    def get_image_url(self, obj):
        if obj.image:
            return obj.image.url
        return None

    def validate_rating(self, value):
        """Rating validation."""
        if not 1 <= value <= 5:
            raise serializers.ValidationError("Rating must be between 1 and 5.")
        return value

    def validate_message(self, value):
        """Message validation."""
        if len(value.strip()) < 10:
            raise serializers.ValidationError("Testimonial message is too short.")
        return value


# ----------------------------------------------------------------------
# 🔹 NOTIFICATIONS - ENHANCED FOR URL INTEGRATION
# ----------------------------------------------------------------------

class NotificationSerializer(EnhancedModelSerializer):
    """Enhanced notification serializer with URL integration."""
    recipient_name = serializers.CharField(source="recipient.get_full_name", read_only=True)
    type_display = serializers.CharField(source="get_type_display", read_only=True)
    priority_display = serializers.CharField(source="get_priority_display", read_only=True)
    api_url = serializers.SerializerMethodField()
    
    class Meta:
        model = Notification
        fields = [
            "id", "type", "type_display", "title", "message", "payload",
            "priority", "priority_display", "read", "read_at", "recipient", "recipient_name",
            "action_url", "expires_at", "created_at", "updated_at", "api_url"
        ]
        read_only_fields = [
            "id", "created_at", "updated_at", "api_url", "type_display", 
            "priority_display", "recipient_name"
        ]

    def get_api_url(self, obj):
        return reverse('notifications-detail', kwargs={'pk': obj.pk})


# ----------------------------------------------------------------------
# 🔹 BLOVION PAGE - ENHANCED FOR URL INTEGRATION
# ----------------------------------------------------------------------

class BlovionPostSerializer(EnhancedModelSerializer):
    """Enhanced Blovion post serializer with URL integration."""
    image_count = serializers.SerializerMethodField()
    categories_data = CategorySerializer(source="categories", many=True, read_only=True)
    category_ids = serializers.PrimaryKeyRelatedField(
        many=True, 
        queryset=Category.objects.all(),
        source='categories',
        write_only=True,
        required=False
    )
    is_recent = serializers.BooleanField(read_only=True)
    api_url = serializers.SerializerMethodField()
    image_urls = serializers.SerializerMethodField()
    
    class Meta:
        model = BlovionPost
        fields = [
            "id", "title", "slug", "caption", "images", "image_count", "image_urls",
            "video_url", "post_type", "categories", "categories_data", "category_ids",
            "published", "published_at", "featured", "social_posted", "social_platforms",
            "read_time", "author", "tools", "dataset_size", "methodology", "complexity",
            "is_recent", "meta_title", "meta_description", "tags", "view_count",
            "created_at", "updated_at", "api_url"
        ]
        read_only_fields = [
            "id", "slug", "view_count", "image_count", "is_recent",
            "created_at", "updated_at", "api_url", "image_urls"
        ]

    def get_api_url(self, obj):
        return reverse('blovion-detail', kwargs={'pk': obj.pk})

    def get_image_count(self, obj):
        """Get count of images."""
        if obj.images:
            return len(obj.images)
        return 0

    def get_image_urls(self, obj):
        """Get list of image URLs."""
        if obj.images:
            return [img.url for img in obj.images]
        return []

    def validate_video_url(self, value):
        """Validate video URL format."""
        if value and not value.startswith(('http://', 'https://')):
            raise serializers.ValidationError("Please provide a valid video URL.")
        return value


# ----------------------------------------------------------------------
# 🔹 AI / SEOVOSOLUTIONS ASSISTANT - ENHANCED FOR URL INTEGRATION
# ----------------------------------------------------------------------

class DocumentSerializer(EnhancedModelSerializer):
    """Enhanced document serializer for AI semantic search with URL integration."""
    source_display = serializers.CharField(source="get_source_display", read_only=True)
    api_url = serializers.SerializerMethodField()
    
    class Meta:
        model = Document
        fields = [
            "id", "title", "text", "source", "source_display",
            "source_id", "chunk_index", "total_chunks", 
            "embedding_vector", "embedding_dim", "language", "is_active",
            "created_at", "updated_at", "api_url"
        ]
        read_only_fields = [
            "id", "embedding_dim", "created_at", "updated_at", "source_display", "api_url"
        ]

    def get_api_url(self, obj):
        # Documents might not have a direct API endpoint, return None
        return None

    def validate_text(self, value):
        """Text validation for documents."""
        if len(value.strip()) < 10:
            raise serializers.ValidationError("Document text is too short.")
        return value


class ConversationLogSerializer(EnhancedModelSerializer):
    """Enhanced conversation log serializer with URL integration."""
    api_url = serializers.SerializerMethodField()
    feedback_url = serializers.SerializerMethodField()
    
    class Meta:
        model = ConversationLog
        fields = [
            "id", "user", "user_identifier", "session_id", "user_message", "bot_reply",
            "intent", "confidence", "sources", "metadata", "user_satisfaction",
            "response_time", "created_at", "updated_at", "api_url", "feedback_url"
        ]
        read_only_fields = [
            "id", "created_at", "updated_at", "api_url", "feedback_url"
        ]

    def get_api_url(self, obj):
        return reverse('conversation-logs-detail', kwargs={'pk': obj.pk})

    def get_feedback_url(self, obj):
        return reverse('assistant-feedback-list')

    def to_representation(self, instance):
        """Enhanced representation with truncation for long replies."""
        data = super().to_representation(instance)
        reply = data.get("bot_reply", "")
        if len(reply) > 800:
            data["bot_reply"] = reply[:797] + "..."
        
        # Format response time
        if data.get("response_time"):
            data["response_time"] = round(data["response_time"], 2)
        
        return data


class AssistantFeedbackSerializer(EnhancedModelSerializer):
    """Enhanced assistant feedback serializer with URL integration."""
    conversation_preview = serializers.CharField(source="conversation.user_message", read_only=True)
    api_url = serializers.SerializerMethodField()
    conversation_url = serializers.SerializerMethodField()
    
    class Meta:
        model = AssistantFeedback
        fields = [
            "id", "conversation", "conversation_preview", "conversation_url", "helpful",
            "rating", "comment", "improvements_suggested", "would_use_again",
            "created_at", "updated_at", "api_url"
        ]
        read_only_fields = [
            "id", "created_at", "updated_at", "conversation_preview", 
            "api_url", "conversation_url"
        ]

    def get_api_url(self, obj):
        return reverse('assistant-feedback-detail', kwargs={'pk': obj.pk})

    def get_conversation_url(self, obj):
        return reverse('conversation-logs-detail', kwargs={'pk': obj.conversation.pk})

    def validate_rating(self, value):
        """Rating validation."""
        if value and not (1 <= value <= 5):
            raise serializers.ValidationError("Rating must be between 1 and 5.")
        return value

    def validate(self, data):
        """Cross-field validation."""
        if data.get('helpful') is False and not data.get('improvements_suggested'):
            raise serializers.ValidationError(
                "Please provide improvement suggestions when marking as not helpful."
            )
        return data


# ----------------------------------------------------------------------
# 🔹 ANALYTICS & DASHBOARD SERIALIZERS - ENHANCED
# ----------------------------------------------------------------------

class AnalyticsSummarySerializer(serializers.Serializer):
    """Serializer for analytics summary data with URL integration."""
    total_registrations = serializers.IntegerField()
    recent_registrations = serializers.IntegerField()
    total_conversations = serializers.IntegerField()
    recent_conversations = serializers.IntegerField()
    total_feedback = serializers.IntegerField()
    average_rating = serializers.FloatField()
    helpful_rate = serializers.FloatField()
    active_services = serializers.IntegerField()
    last_updated = serializers.DateTimeField()
    
    # URLs for detailed data
    registrations_url = serializers.CharField(default=reverse('training-list'))
    conversations_url = serializers.CharField(default=reverse('conversation-logs-list'))
    feedback_url = serializers.CharField(default=reverse('assistant-feedback-list'))


class ServiceStatsSerializer(serializers.Serializer):
    """Serializer for AI service statistics."""
    service = serializers.CharField()
    success = serializers.IntegerField()
    failures = serializers.IntegerField()
    success_rate = serializers.FloatField()
    last_used = serializers.DateTimeField(allow_null=True)


# ----------------------------------------------------------------------
# 🔹 BULK OPERATIONS SERIALIZER
# ----------------------------------------------------------------------

class BulkOperationSerializer(serializers.Serializer):
    """Serializer for bulk operations endpoint."""
    operation = serializers.ChoiceField(choices=[
        'export_training_data',
        'export_events_data', 
        'export_posts_data',
        'cleanup_old_data'
    ])
    data = serializers.ListField(child=serializers.DictField(), required=False)
    
    def validate(self, data):
        """Validate bulk operation data."""
        operation = data.get('operation')
        if operation.startswith('export') and not self.context.get('request').user.is_staff:
            raise serializers.ValidationError("Admin access required for export operations.")
        return data


# ----------------------------------------------------------------------
# 🔹 FILE UPLOAD SERIALIZER FOR ENHANCED FILE UPLOAD VIEW
# ----------------------------------------------------------------------

class EnhancedFileUploadResponseSerializer(serializers.Serializer):
    """Serializer for enhanced file upload response."""
    success = serializers.BooleanField()
    message = serializers.CharField()
    file = serializers.DictField()
    metadata = serializers.DictField()


# ----------------------------------------------------------------------
# 🔹 HEALTH CHECK SERIALIZER
# ----------------------------------------------------------------------

class HealthCheckSerializer(serializers.Serializer):
    """Serializer for health check endpoint."""
    status = serializers.CharField()
    timestamp = serializers.DateTimeField()
    service = serializers.CharField()
    version = serializers.CharField()
    environment = serializers.CharField()
    checks = serializers.DictField()
    system_metrics = serializers.DictField(required=False)


# ----------------------------------------------------------------------
# 🔹 EXPORT ALL SERIALIZERS
# ----------------------------------------------------------------------

__all__ = [
    'EnhancedModelSerializer',
    'CategorySerializer', 'SiteConfigSerializer', 'SystemMetricSerializer', 
    'PageViewSerializer', 'ContactSerializer', 'TrainingProgramSerializer',
    'TrainingRegistrationSerializer', 'FileUploadSerializer', 'PostSerializer', 
    'EventSerializer', 'TestimonialSerializer', 'NotificationSerializer', 
    'BlovionPostSerializer', 'DocumentSerializer', 'ConversationLogSerializer', 
    'AssistantFeedbackSerializer', 'AnalyticsSummarySerializer', 'ServiceStatsSerializer', 
    'BulkOperationSerializer', 'EnhancedFileUploadResponseSerializer', 'HealthCheckSerializer'
]