# website/signals.py
import logging
import threading
from django.db.models.signals import post_save
from django.dispatch import receiver
from asgiref.sync import async_to_sync
from channels.layers import get_channel_layer
from django.conf import settings

from .models import Contact, TrainingRegistration, Post, Notification, BlovionPost
from .serializers import (
    ContactSerializer,
    TrainingRegistrationSerializer,
    PostSerializer,
    BlovionPostSerializer,
)

logger = logging.getLogger(__name__)
channel_layer = get_channel_layer()


# ------------------------------------------------------------------------------
# 🔄 Utility: Send WebSocket notification to group
# ------------------------------------------------------------------------------
def notify_group(event_type, data):
    """Notify connected clients of a backend event in real time."""
    try:
        async_to_sync(channel_layer.group_send)(
            "notifications",
            {
                "type": "send_notification",
                "content": {"type": event_type, "data": data},
            },
        )
    except Exception as e:
        logger.error(f"WebSocket notify_group failed: {e}")


# ------------------------------------------------------------------------------
# 🧠 Generic helper for dashboard updates
# ------------------------------------------------------------------------------
def push_realtime_update(event_type, payload):
    """Send updates to all connected dashboard clients."""
    try:
        async_to_sync(channel_layer.group_send)(
            "dashboard_updates",
            {"type": "send_update", "data": {"event": event_type, "payload": payload}},
        )
    except Exception as e:
        logger.warning(f"Realtime dashboard update failed: {e}")


# ------------------------------------------------------------------------------
# 📨 Signal Handlers
# ------------------------------------------------------------------------------

@receiver(post_save, sender=Contact)
def contact_created(sender, instance, created, **kwargs):
    """Handle new contact form submissions."""
    if created:
        try:
            data = ContactSerializer(instance).data
            notify_group("contact_created", data)
            push_realtime_update("contact_created", data)
            logger.info(f"📧 New contact created: {instance.name} <{instance.email}>")
        except Exception as e:
            logger.error(f"❌ Contact signal error: {e}")


@receiver(post_save, sender=TrainingRegistration)
def registration_created(sender, instance, created, **kwargs):
    """Handle new training registrations."""
    if created:
        try:
            data = TrainingRegistrationSerializer(instance).data
            notify_group("registration_created", data)
            push_realtime_update("training_registration_created", data)
            logger.info(f"🎓 New training registration: {instance.full_name} for {instance.program}")
        except Exception as e:
            logger.error(f"❌ Training registration signal error: {e}")


# ------------------------------------------------------------------------------
# 📰 Post Created (Blog Post)
# ------------------------------------------------------------------------------
@receiver(post_save, sender=Post)
def post_created(sender, instance, created, **kwargs):
    """
    When a new blog post is created and published:
    - Send real-time update to clients
    - Broadcast to all social media (if API keys are available)
    """
    try:
        if created and instance.published:
            data = PostSerializer(instance).data

            # Notify connected users
            notify_group("post_published", data)
            push_realtime_update("post_published", data)

            # Post to social media asynchronously
            threading.Thread(
                target=auto_publish_social_media, args=(instance,), daemon=True
            ).start()

            logger.info(f"✅ Blog post published and broadcasted: {instance.title}")
    except Exception as e:
        logger.error(f"❌ Error processing post_created signal: {e}")


# ------------------------------------------------------------------------------
# 📱 Blovion Post Created
# ------------------------------------------------------------------------------
@receiver(post_save, sender=BlovionPost)
def blovion_post_created(sender, instance, created, **kwargs):
    """
    When a new Blovion post is created and published:
    - Send real-time update to clients
    - Broadcast to social media
    """
    try:
        if created and instance.published:
            data = BlovionPostSerializer(instance).data

            # Notify connected users
            notify_group("blovion_post_published", data)
            push_realtime_update("blovion_post_published", data)

            # Post to social media asynchronously
            threading.Thread(
                target=auto_publish_social_media, args=(instance,), daemon=True
            ).start()

            logger.info(f"✅ Blovion post published and broadcasted: {instance.title}")
    except Exception as e:
        logger.error(f"❌ Error processing blovion_post_created signal: {e}")


# ------------------------------------------------------------------------------
# 🔔 Notification Created
# ------------------------------------------------------------------------------
@receiver(post_save, sender=Notification)
def notification_created(sender, instance, created, **kwargs):
    """Handle new notification creation."""
    if created:
        try:
            # Send real-time notification to the specific user
            user_group = f"user_{instance.recipient.id}"
            async_to_sync(channel_layer.group_send)(
                user_group,
                {
                    "type": "send_notification",
                    "content": {
                        "type": "new_notification",
                        "data": {
                            "id": instance.id,
                            "title": instance.title,
                            "message": instance.message,
                            "type": instance.type,
                            "priority": instance.priority,
                            "created_at": instance.created_at.isoformat()
                        }
                    }
                },
            )
            logger.info(f"🔔 Notification sent to user {instance.recipient.id}: {instance.title}")
        except Exception as e:
            logger.error(f"❌ Notification signal error: {e}")


# ------------------------------------------------------------------------------
# 🌍 Social Media Cross-Posting
# ------------------------------------------------------------------------------
def auto_publish_social_media(post):
    """
    Automatically publish the given post to connected social media platforms.
    Safely handles missing social media utilities.
    """
    try:
        # Try to import social media utility
        from website.utils import publish_to_social_media
        # Run safely in thread
        publish_to_social_media(post)
        logger.info(f"📱 Social media auto-publish initiated for: {post.title}")

    except ImportError:
        logger.warning("⚠️ Social media utilities not available - skipping auto-publish")
    except Exception as e:
        logger.error(f"❌ Failed social media auto-publish for '{post.title}': {e}")


# ------------------------------------------------------------------------------
# 🎯 System Metrics Tracking
# ------------------------------------------------------------------------------
@receiver(post_save, sender=TrainingRegistration)
def track_registration_metrics(sender, instance, created, **kwargs):
    """Track training registration metrics for analytics."""
    if created:
        try:
            from .models import SystemMetric
            SystemMetric.objects.create(
                metric_type="training_registration",
                value=1,
                metadata={
                    "program": instance.program,
                    "source": instance.referral_source or "direct",
                    "timestamp": instance.created_at.isoformat()
                }
            )
        except Exception as e:
            logger.warning(f"⚠️ Could not track registration metrics: {e}")


@receiver(post_save, sender=Contact)
def track_contact_metrics(sender, instance, created, **kwargs):
    """Track contact form metrics for analytics."""
    if created:
        try:
            from .models import SystemMetric
            SystemMetric.objects.create(
                metric_type="contact_submission",
                value=1,
                metadata={
                    "contact_type": instance.contact_type,
                    "priority": instance.priority,
                    "timestamp": instance.created_at.isoformat()
                }
            )
        except Exception as e:
            logger.warning(f"⚠️ Could not track contact metrics: {e}")