# ---------------------------------------------------
# OvoSolutions Automated Data Updater & Sync Manager
# ---------------------------------------------------
# This module manages automated backend refresh tasks for:
# - Testimonials (auto-generated or new entries)
# - Events (updated schedules or new posts)
# - Trainings (rotating demo sessions)
# - Notifications (system + user feed)
#
# Runs automatically twice per day (every 12 hours)
# to keep backend and frontend data synchronized.
# ---------------------------------------------------

import random
import logging
from datetime import datetime, timedelta
from django.db import transaction
from django.utils.timezone import now

# APScheduler (safe fallback for periodic updates)
try:
    from apscheduler.schedulers.background import BackgroundScheduler
    from django_apscheduler.jobstores import DjangoJobStore, register_events
except ImportError:
    BackgroundScheduler = None

# Celery integration
try:
    from ovosolutions_backend import celery_app
except ImportError:
    celery_app = None

# Models
from .models import Testimonial, Notification, Event, TrainingProgram

logger = logging.getLogger(__name__)


# --------------------------------------------------------------------
# 🔹 Main Auto-Update Task
# --------------------------------------------------------------------
def sync_frontend_data():
    """
    Performs simulated content updates and auto-syncs data
    between backend and frontend every 12 hours.
    Safe for both Celery task execution and manual triggering.
    """
    try:
        logger.info("[AutoUpdate] Running scheduled data sync at %s", datetime.now())

        names = [
            "David", "Eunice", "Nancy", "Kiprotich", "Okoth",
            "Francis", "Maxwel", "Emmanuel", "Godwill", "Janet"
        ]
        messages = [
            "Fantastic mentorship and career guidance!",
            "The analytics bootcamp was deeply insightful.",
            "Loved the real-world case studies.",
            "SPSS and Python integration was top-notch!",
            "SeovoSolutions exceeded my expectations!",
        ]

        # ✅ 1. Occasionally add a new Testimonial
        if random.random() < 0.5:
            with transaction.atomic():
                name = random.choice(names)
                message = random.choice(messages)
                Testimonial.objects.create(
                    name=name,
                    message=message,
                    approved=False,
                    rating=random.randint(4, 5),
                )
                logger.info(f"[AutoUpdate] Added testimonial by {name}")

        # ✅ 2. Occasionally add a new TrainingProgram
        if random.random() < 0.4:
            with transaction.atomic():
                new_training = TrainingProgram.objects.create(
                    title=f"Data Science Bootcamp {random.randint(100,999)}",
                    description="Updated 3-week data science mentorship focusing on ML and Python.",
                    instructor="SeovoSolutions Mentorship Team",
                    capacity=random.randint(10, 30),
                    is_active=True,
                    registration_open=True,
                )
                logger.info(f"[AutoUpdate] Added training: {new_training.title}")

        # ✅ 3. Occasionally add an Event
        if random.random() < 0.4:
            with transaction.atomic():
                event = Event.objects.create(
                    title=f"AI Innovation Forum {random.randint(10,99)}",
                    description="A bi-annual forum on AI innovations and emerging technologies.",
                    start_time=now() + timedelta(days=random.randint(5, 30)),
                    location="SeovoSolutions HQ, Nairobi",
                    capacity=random.randint(50, 150),
                    is_virtual=random.choice([True, False]),
                )
                logger.info(f"[AutoUpdate] New event created: {event.title}")

        # ✅ 4. Always log a Notification
        Notification.objects.create(
            type="info",
            recipient=None,
            payload={
                "title": "System Auto-Update",
                "message": f"Data synchronized successfully on {now().strftime('%Y-%m-%d %H:%M')}."
            },
            read=False,
        )
        logger.info("[AutoUpdate] Notifications refreshed successfully.")
        logger.info("[AutoUpdate] ✅ Sync completed successfully.")

    except Exception as e:
        logger.exception(f"[AutoUpdate] ❌ Sync failed: {e}")


# --------------------------------------------------------------------
# 🔹 Celery Task Wrapper (if Celery is installed)
# --------------------------------------------------------------------
if celery_app:
    @celery_app.task(name="website.tasks.sync_frontend_data")
    def sync_frontend_data_task():
        """Celery-compatible wrapper for periodic scheduling."""
        sync_frontend_data()


# --------------------------------------------------------------------
# 🔹 APScheduler Fallback Scheduler
# --------------------------------------------------------------------
_scheduler_instance = None  # Prevent multiple schedulers on reload


def start_scheduler():
    """
    Starts APScheduler (if Celery Beat is unavailable).
    Runs every 12 hours to update testimonials, events, etc.
    """
    global _scheduler_instance

    if _scheduler_instance or not BackgroundScheduler:
        if not BackgroundScheduler:
            logger.warning("[AutoUpdate] APScheduler not installed — skipping fallback scheduler.")
        return

    try:
        scheduler = BackgroundScheduler(timezone="Africa/Nairobi")
        scheduler.add_jobstore(DjangoJobStore(), "default")

        scheduler.add_job(
            sync_frontend_data,
            trigger="interval",
            hours=12,
            id="website_data_auto_updater",
            replace_existing=True,
        )

        register_events(scheduler)
        scheduler.start()
        _scheduler_instance = scheduler
        logger.info("[AutoUpdate] APScheduler started successfully (every 12 hours).")

    except Exception as e:
        logger.exception(f"[AutoUpdate] Failed to start APScheduler: {e}")


# --------------------------------------------------------------------
# 🔹 Auto-trigger Scheduler on Import
# --------------------------------------------------------------------
try:
    if celery_app:
        # Schedule first Celery run (startup trigger)
        eta = now() + timedelta(seconds=15)
        celery_app.send_task("website.tasks.sync_frontend_data", eta=eta)
        logger.info("[AutoUpdate] Celery initial sync task scheduled successfully.")
    else:
        # Start APScheduler fallback
        start_scheduler()
except Exception as e:
    logger.warning(f"[AutoUpdate] Scheduler setup failed: {e}")
