# website/tests.py
import os
import json
from datetime import timedelta
from unittest.mock import patch, MagicMock

from django.test import TestCase, Client
from django.urls import reverse
from django.contrib.auth import get_user_model
from django.utils import timezone
from django.core.files.uploadedfile import SimpleUploadedFile
from django.core import mail

from rest_framework import status
from rest_framework.test import APITestCase, APIClient

from .models import (
    Contact, TrainingProgram, TrainingRegistration, FileUpload, Post, Event,
    Testimonial, Notification, BlovionPost, Category, ConversationLog,
    AssistantFeedback, SystemMetric, SiteConfig
)
from .serializers import (
    ContactSerializer, TrainingProgramSerializer, TrainingRegistrationSerializer,
    PostSerializer, EventSerializer, TestimonialSerializer, BlovionPostSerializer
)

User = get_user_model()


# ==============================================================================
# 🔧 TEST SETUP & UTILITIES
# ==============================================================================

class BaseTestCase(APITestCase):
    """Base test case with common setup and utilities."""
    
    def setUp(self):
        """Set up test data."""
        self.client = APIClient()
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )
        self.admin_user = User.objects.create_superuser(
            username='admin',
            email='admin@example.com',
            password='adminpass123'
        )
        
        # Create test category
        self.category = Category.objects.create(
            name='Technology',
            slug='technology',
            description='Tech related content',
            color='#3B82F6'
        )
        
        # Create test training program
        self.training_program = TrainingProgram.objects.create(
            title='Web Development Bootcamp',
            description='Learn full-stack web development',
            program_type='bootcamp',
            level='beginner',
            start_date=timezone.now().date() + timedelta(days=30),
            end_date=timezone.now().date() + timedelta(days=120),
            capacity=50,
            price=999.00,
            registration_open=True,
            featured=True
        )
        
        # Create test contact
        self.contact = Contact.objects.create(
            name='John Doe',
            email='john@example.com',
            subject='Test Inquiry',
            message='This is a test message'
        )


# ==============================================================================
# 🧪 MODEL TESTS
# ==============================================================================

class ModelTests(BaseTestCase):
    """Test model functionality and methods."""
    
    def test_contact_creation(self):
        """Test contact model creation and methods."""
        contact = Contact.objects.create(
            name='Jane Smith',
            email='jane@example.com',
            subject='Partnership Inquiry',
            message='Interested in partnership opportunities'
        )
        
        self.assertEqual(contact.name, 'Jane Smith')
        self.assertEqual(contact.email, 'jane@example.com')
        self.assertFalse(contact.responded)
        
        # Test frontend dict method
        frontend_data = contact.to_frontend_dict()
        self.assertIn('id', frontend_data)
        self.assertIn('name', frontend_data)
        self.assertIn('email', frontend_data)
        
        # Test mark as responded
        contact.mark_responded(self.user, 'Test response')
        self.assertTrue(contact.responded)
        self.assertEqual(contact.response_notes, 'Test response')
    
    def test_training_program_creation(self):
        """Test training program model creation and methods."""
        program = TrainingProgram.objects.create(
            title='Data Science Course',
            description='Learn data science and machine learning',
            program_type='course',
            level='intermediate',
            start_date=timezone.now().date() + timedelta(days=15),
            end_date=timezone.now().date() + timedelta(days=90),
            capacity=30,
            price=799.00
        )
        
        self.assertEqual(program.title, 'Data Science Course')
        self.assertTrue(program.is_upcoming())
        self.assertTrue(program.is_registration_open())
        self.assertEqual(program.available_spots(), 30)
        
        # Test frontend dict method
        frontend_data = program.to_frontend_dict()
        self.assertIn('id', frontend_data)
        self.assertIn('title', frontend_data)
        self.assertIn('is_upcoming', frontend_data)
        self.assertIn('available_spots', frontend_data)
    
    def test_training_registration_creation(self):
        """Test training registration model creation and methods."""
        registration = TrainingRegistration.objects.create(
            full_name='Alice Johnson',
            email='alice@example.com',
            program=self.training_program,
            phone='+1234567890',
            company='Tech Corp'
        )
        
        self.assertEqual(registration.full_name, 'Alice Johnson')
        self.assertEqual(registration.program, self.training_program)
        self.assertEqual(registration.status, 'pending')
        self.assertIsNotNone(registration.registration_code)
        
        # Test frontend dict method
        frontend_data = registration.to_frontend_dict()
        self.assertIn('id', frontend_data)
        self.assertIn('full_name', frontend_data)
        self.assertIn('program_title', frontend_data)
        self.assertIn('registration_code', frontend_data)
    
    def test_post_creation(self):
        """Test blog post model creation and methods."""
        post = Post.objects.create(
            title='Test Blog Post',
            content='This is a test blog post content',
            author=self.user,
            published=True
        )
        
        self.assertEqual(post.title, 'Test Blog Post')
        self.assertTrue(post.published)
        self.assertGreater(post.word_count, 0)
        
        # Test frontend dict method
        frontend_data = post.to_frontend_dict()
        self.assertIn('id', frontend_data)
        self.assertIn('title', frontend_data)
        self.assertIn('author', frontend_data)
        self.assertIn('word_count', frontend_data)
    
    def test_blovion_post_creation(self):
        """Test Blovion post model creation and methods."""
        blovion_post = BlovionPost.objects.create(
            title='Test Blovion Post',
            caption='This is a test Blovion post',
            author='SeovoSolutions Team',
            published=True
        )
        
        self.assertEqual(blovion_post.title, 'Test Blovion Post')
        self.assertTrue(blovion_post.published)
        
        # Test frontend dict method
        frontend_data = blovion_post.to_frontend_dict()
        self.assertIn('id', frontend_data)
        self.assertIn('title', frontend_data)
        self.assertIn('author', frontend_data)
        self.assertIn('image_count', frontend_data)


# ==============================================================================
# 🌐 API ENDPOINT TESTS
# ==============================================================================

class APITests(BaseTestCase):
    """Test API endpoints."""
    
    def test_api_root(self):
        """Test API root endpoint."""
        url = reverse('api-root')
        response = self.client.get(url)
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertIn('service', response.data)
        self.assertIn('endpoints', response.data)
        self.assertIn('features', response.data)
    
    def test_health_check(self):
        """Test health check endpoint."""
        url = reverse('health-check')
        response = self.client.get(url)
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertIn('status', response.data)
        self.assertIn('services', response.data)
        self.assertIn('timestamp', response.data)
    
    def test_contact_create(self):
        """Test contact form submission."""
        url = reverse('contacts-list')
        data = {
            'name': 'Test User',
            'email': 'test@example.com',
            'subject': 'Test Subject',
            'message': 'This is a test message for contact form.'
        }
        
        response = self.client.post(url, data, format='json')
        
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Contact.objects.count(), 2)  # One from setUp + this one
        self.assertEqual(Contact.objects.last().name, 'Test User')
    
    def test_training_registration_create(self):
        """Test training registration submission."""
        url = reverse('training-list')
        data = {
            'full_name': 'Test Student',
            'email': 'student@example.com',
            'phone': '+254712345678',
            'program': self.training_program.id,
            'company': 'Test Company',
            'experience_level': 'beginner',
            'learning_goals': 'Learn web development skills'
        }
        
        response = self.client.post(url, data, format='json')
        
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(TrainingRegistration.objects.count(), 1)
        self.assertEqual(TrainingRegistration.objects.last().full_name, 'Test Student')
    
    def test_training_programs_list(self):
        """Test training programs listing."""
        url = reverse('training-programs-list')
        response = self.client.get(url)
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertGreater(len(response.data), 0)
    
    def test_posts_list(self):
        """Test blog posts listing."""
        # Create a test post
        Post.objects.create(
            title='Test Post',
            content='Test content',
            published=True
        )
        
        url = reverse('posts-list')
        response = self.client.get(url)
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertGreater(len(response.data), 0)
    
    def test_events_list(self):
        """Test events listing."""
        # Create a test event
        Event.objects.create(
            title='Test Event',
            description='Test event description',
            start_time=timezone.now() + timedelta(days=1),
            end_time=timezone.now() + timedelta(days=1, hours=2)
        )
        
        url = reverse('events-list')
        response = self.client.get(url)
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertGreater(len(response.data), 0)
    
    def test_testimonials_list(self):
        """Test testimonials listing."""
        # Create a test testimonial
        Testimonial.objects.create(
            name='Test Client',
            company='Test Company',
            message='Great service!',
            rating=5,
            approved=True
        )
        
        url = reverse('testimonials-list')
        response = self.client.get(url)
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertGreater(len(response.data), 0)


# ==============================================================================
# 🤖 AI CHAT TESTS
# ==============================================================================

class AIChatTests(BaseTestCase):
    """Test AI chat functionality."""
    
    @patch('website.views.ai_manager.get_ai_response')
    def test_ai_chat_success(self, mock_ai_response):
        """Test successful AI chat interaction."""
        mock_ai_response.return_value = ("Hello! I'm your AI assistant.", "gemini")
        
        url = reverse('ai-chat')
        data = {
            'message': 'Hello, can you help me?',
            'user_id': 'test_user_123'
        }
        
        response = self.client.post(url, data, format='json')
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertIn('reply', response.data)
        self.assertIn('service_used', response.data)
        self.assertEqual(response.data['service_used'], 'gemini')
    
    @patch('website.views.ai_manager.get_ai_response')
    def test_ai_chat_fallback(self, mock_ai_response):
        """Test AI chat with fallback response."""
        mock_ai_response.return_value = ("I'd love to help! Contact us for more info.", "fallback")
        
        url = reverse('ai-chat')
        data = {
            'message': 'Tell me about your services'
        }
        
        response = self.client.post(url, data, format='json')
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertIn('reply', response.data)
        self.assertEqual(response.data['service_used'], 'fallback')
    
    def test_ai_chat_empty_message(self):
        """Test AI chat with empty message."""
        url = reverse('ai-chat')
        data = {
            'message': ''
        }
        
        response = self.client.post(url, data, format='json')
        
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertIn('error', response.data)


# ==============================================================================
# 📊 STATISTICS & ANALYTICS TESTS
# ==============================================================================

class StatisticsTests(BaseTestCase):
    """Test statistics and analytics endpoints."""
    
    def test_enhanced_stats(self):
        """Test enhanced statistics endpoint."""
        url = reverse('stats')
        response = self.client.get(url)
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertIn('core_metrics', response.data)
        self.assertIn('ai_metrics', response.data)
        
        core_metrics = response.data['core_metrics']
        self.assertIn('experience', core_metrics)
        self.assertIn('projects', core_metrics)
        self.assertIn('students', core_metrics)
        self.assertIn('trainings', core_metrics)
        self.assertIn('clients', core_metrics)
    
    def test_service_status(self):
        """Test service status endpoint."""
        url = reverse('service-status')
        response = self.client.get(url)
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertIn('services', response.data)
        self.assertIn('feature_flags', response.data)
        self.assertIn('limits', response.data)


# ==============================================================================
# 📁 FILE UPLOAD TESTS
# ==============================================================================

class FileUploadTests(BaseTestCase):
    """Test file upload functionality."""
    
    def test_file_upload_success(self):
        """Test successful file upload."""
        url = reverse('file-upload')
        
        # Create a test file
        test_file = SimpleUploadedFile(
            "test_image.jpg",
            b"file_content",
            content_type="image/jpeg"
        )
        
        data = {
            'file': test_file,
            'title': 'Test File',
            'description': 'Test file description',
            'category': 'image'
        }
        
        response = self.client.post(url, data, format='multipart')
        
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertTrue(response.data['success'])
        self.assertIn('file', response.data)
        self.assertIn('metadata', response.data)
    
    def test_file_upload_invalid_type(self):
        """Test file upload with invalid file type."""
        url = reverse('file-upload')
        
        test_file = SimpleUploadedFile(
            "test.exe",
            b"file_content",
            content_type="application/octet-stream"
        )
        
        data = {'file': test_file}
        
        response = self.client.post(url, data, format='multipart')
        
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertIn('error', response.data)
        self.assertEqual(response.data['code'], 'DANGEROUS_FILE_TYPE')
    
    def test_file_upload_no_file(self):
        """Test file upload without file."""
        url = reverse('file-upload')
        data = {}
        
        response = self.client.post(url, data, format='multipart')
        
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertIn('error', response.data)
        self.assertEqual(response.data['code'], 'NO_FILE')


# ==============================================================================
# 🔐 AUTHENTICATION & PERMISSION TESTS
# ==============================================================================

class AuthenticationTests(BaseTestCase):
    """Test authentication and permissions."""
    
    def test_bulk_operations_admin_required(self):
        """Test that bulk operations require admin access."""
        url = reverse('bulk-operations')
        data = {
            'operation': 'export_training_data'
        }
        
        # Test without authentication
        response = self.client.post(url, data, format='json')
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)
        
        # Test with regular user
        self.client.force_authenticate(user=self.user)
        response = self.client.post(url, data, format='json')
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)
        
        # Test with admin user
        self.client.force_authenticate(user=self.admin_user)
        response = self.client.post(url, data, format='json')
        self.assertEqual(response.status_code, status.HTTP_200_OK)


# ==============================================================================
# 🔄 SERIALIZER TESTS
# ==============================================================================

class SerializerTests(BaseTestCase):
    """Test serializer validation and functionality."""
    
    def test_contact_serializer_validation(self):
        """Test contact serializer validation."""
        # Valid data
        valid_data = {
            'name': 'Test User',
            'email': 'test@example.com',
            'subject': 'Test Subject',
            'message': 'This is a proper test message with enough content.'
        }
        
        serializer = ContactSerializer(data=valid_data)
        self.assertTrue(serializer.is_valid())
        
        # Invalid email
        invalid_email_data = valid_data.copy()
        invalid_email_data['email'] = 'invalid-email'
        serializer = ContactSerializer(data=invalid_email_data)
        self.assertFalse(serializer.is_valid())
        self.assertIn('email', serializer.errors)
        
        # Short message
        short_message_data = valid_data.copy()
        short_message_data['message'] = 'Short'
        serializer = ContactSerializer(data=short_message_data)
        self.assertFalse(serializer.is_valid())
        self.assertIn('message', serializer.errors)
    
    def test_training_registration_serializer_validation(self):
        """Test training registration serializer validation."""
        valid_data = {
            'full_name': 'Test Student',
            'email': 'student@example.com',
            'program': self.training_program.id
        }
        
        serializer = TrainingRegistrationSerializer(data=valid_data)
        self.assertTrue(serializer.is_valid())
        
        # Invalid email
        invalid_email_data = valid_data.copy()
        invalid_email_data['email'] = 'invalid-email'
        serializer = TrainingRegistrationSerializer(data=invalid_email_data)
        self.assertFalse(serializer.is_valid())
        self.assertIn('email', serializer.errors)


# ==============================================================================
# 📈 MODEL INTEGRATION TESTS
# ==============================================================================

class ModelIntegrationTests(BaseTestCase):
    """Test model integration and relationships."""
    
    def test_training_program_enrollment_count(self):
        """Test training program enrollment count updates."""
        initial_count = self.training_program.current_enrollment
        
        # Create a registration
        TrainingRegistration.objects.create(
            full_name='Student One',
            email='student1@example.com',
            program=self.training_program,
            status='confirmed'
        )
        
        # Refresh program from database
        self.training_program.refresh_from_db()
        self.assertEqual(self.training_program.current_enrollment, initial_count + 1)
    
    def test_post_category_relationship(self):
        """Test post-category relationship."""
        post = Post.objects.create(
            title='Test Post with Category',
            content='Test content',
            author=self.user
        )
        post.categories.add(self.category)
        
        self.assertEqual(post.categories.count(), 1)
        self.assertEqual(post.categories.first(), self.category)


# ==============================================================================
# 🎯 ERROR HANDLING TESTS
# ==============================================================================

class ErrorHandlingTests(BaseTestCase):
    """Test error handling and edge cases."""
    
    def test_404_handler(self):
        """Test custom 404 error handler."""
        url = '/api/nonexistent-endpoint/'
        response = self.client.get(url)
        
        self.assertEqual(response.status_code, status.HTTP_404_NOT_FOUND)
        self.assertIn('error', response.data)
        self.assertIn('suggestions', response.data)
    
    def test_invalid_training_program_registration(self):
        """Test registration for non-existent training program."""
        url = reverse('training-list')
        data = {
            'full_name': 'Test Student',
            'email': 'student@example.com',
            'program': 9999  # Non-existent program ID
        }
        
        response = self.client.post(url, data, format='json')
        
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)


# ==============================================================================
# 🧹 CLEANUP & MAINTENANCE TESTS
# ==============================================================================

class CleanupTests(BaseTestCase):
    """Test data cleanup and maintenance operations."""
    
    def test_bulk_operations_cleanup(self):
        """Test bulk cleanup operations."""
        # Create old conversation logs
        old_date = timezone.now() - timedelta(days=100)
        ConversationLog.objects.create(
            user_identifier='test_user',
            user_message='Old message',
            bot_reply='Old reply',
            created_at=old_date
        )
        
        initial_count = ConversationLog.objects.count()
        
        # Perform cleanup operation (requires admin auth)
        self.client.force_authenticate(user=self.admin_user)
        url = reverse('bulk-operations')
        data = {
            'operation': 'cleanup_old_data'
        }
        
        response = self.client.post(url, data, format='json')
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertIn('deleted_records', response.data)
        
        # Verify cleanup occurred
        final_count = ConversationLog.objects.count()
        self.assertLess(final_count, initial_count)


# ==============================================================================
# 🚀 PERFORMANCE & SCALING TESTS
# ==============================================================================

class PerformanceTests(BaseTestCase):
    """Test performance and scaling considerations."""
    
    def test_multiple_registrations_performance(self):
        """Test performance with multiple registrations."""
        # Create multiple registrations
        for i in range(10):
            TrainingRegistration.objects.create(
                full_name=f'Student {i}',
                email=f'student{i}@example.com',
                program=self.training_program
            )
        
        # Test listing performance
        url = reverse('training-list')
        
        import time
        start_time = time.time()
        response = self.client.get(url)
        end_time = time.time()
        
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        
        # Ensure response time is reasonable (under 1 second)
        response_time = end_time - start_time
        self.assertLess(response_time, 1.0)


# ==============================================================================
# 🎉 RUN ALL TESTS
# ==============================================================================

if __name__ == '__main__':
    import django
    django.setup()
    
    # Run specific test classes
    import unittest
    unittest.main()