import os
import logging
from datetime import timedelta
from django.urls import path, include, re_path
from django.http import JsonResponse
from django.conf import settings
from django.utils import timezone
from django.views.static import serve
from rest_framework import status
from rest_framework.decorators import api_view, permission_classes
from rest_framework.response import Response
from rest_framework.routers import DefaultRouter
from rest_framework.views import APIView
from rest_framework.parsers import MultiPartParser, FormParser
from rest_framework.permissions import AllowAny, IsAuthenticated, IsAdminUser

logger = logging.getLogger(__name__)

# ---------------------------------------------------------------------
# 🔹 SAFE VIEW IMPORTS WITH ERROR HANDLING
# ---------------------------------------------------------------------
try:
    # ✅ Import all enhanced ViewSets and API Views from improved views
    from .views import (
        ContactViewSet,
        EnhancedTrainingViewSet,  # Updated enhanced training viewset
        TrainingProgramViewSet,   # New training programs viewset
        EventViewSet,
        TestimonialViewSet,
        PostViewSet,
        NotificationViewSet,
        BlovionPostViewSet,
        FileUploadViewSet,
        EnhancedAIChatView,           # Updated enhanced AI chat
        EnhancedAssistantFeedbackView, # Updated enhanced feedback
        EnhancedStatsView,        # Updated enhanced stats
        HealthCheckView,          # New health check view
        api_root,                 # API root function
    )
    VIEWS_IMPORT_SUCCESS = True
    logger.info("✅ Successfully imported all views")
except ImportError as e:
    VIEWS_IMPORT_SUCCESS = False
    logger.error(f"❌ Failed to import views: {e}")
    # Create fallback views to prevent startup crashes
    class FallbackView(APIView):
        permission_classes = [AllowAny]
        def get(self, request):
            return Response({"error": "Service temporarily unavailable - views import failed"}, status=503)

    # Create fallback instances
    ContactViewSet = type('ContactViewSet', (FallbackView,), {})
    EnhancedTrainingViewSet = type('EnhancedTrainingViewSet', (FallbackView,), {})
    TrainingProgramViewSet = type('TrainingProgramViewSet', (FallbackView,), {})
    EventViewSet = type('EventViewSet', (FallbackView,), {})
    TestimonialViewSet = type('TestimonialViewSet', (FallbackView,), {})
    PostViewSet = type('PostViewSet', (FallbackView,), {})
    NotificationViewSet = type('NotificationViewSet', (FallbackView,), {})
    BlovionPostViewSet = type('BlovionPostViewSet', (FallbackView,), {})
    FileUploadViewSet = type('FileUploadViewSet', (FallbackView,), {})
    EnhancedAIChatView = FallbackView
    EnhancedAssistantFeedbackView = FallbackView
    EnhancedStatsView = FallbackView
    HealthCheckView = FallbackView
    
    def api_root(request):
        return JsonResponse({
            "error": "Service temporarily unavailable",
            "message": "Views import failed during startup",
            "timestamp": timezone.now().isoformat()
        }, status=503)

# ---------------------------------------------------------------------
# 🔹 ENHANCED FILE UPLOAD VIEW (Safe and Robust)
# ---------------------------------------------------------------------
class EnhancedFileUploadView(APIView):
    """
    Enhanced file upload handler with comprehensive security, validation, and progress tracking.
    """
    parser_classes = (MultiPartParser, FormParser)
    permission_classes = [AllowAny]

    def post(self, request, *args, **kwargs):
        """
        Handle file upload with comprehensive validation and security checks
        """
        try:
            file_obj = request.data.get("file")
            if not file_obj:
                return Response(
                    {
                        "success": False,
                        "error": "No file provided.",
                        "code": "NO_FILE"
                    },
                    status=status.HTTP_400_BAD_REQUEST,
                )

            # Enhanced file validation
            validation_result = self.validate_file(file_obj)
            if not validation_result["valid"]:
                return Response(
                    {
                        "success": False,
                        "error": validation_result["error"],
                        "code": validation_result["code"]
                    },
                    status=status.HTTP_400_BAD_REQUEST,
                )

            # Generate safe filename and path
            file_info = self.generate_safe_filepath(file_obj.name)
            
            # Save file with progress tracking
            save_result = self.save_file_safely(file_obj, file_info["full_path"])
            if not save_result["success"]:
                return Response(
                    {
                        "success": False,
                        "error": save_result["error"],
                        "code": "SAVE_ERROR"
                    },
                    status=status.HTTP_500_INTERNAL_SERVER_ERROR,
                )

            # Create database record
            db_record = self.create_file_record(file_obj, file_info, request.user)
            
            logger.info(f"✅ File '{file_info['safe_name']}' uploaded successfully by {request.user or 'anonymous'}")
            
            return Response(
                {
                    "success": True,
                    "message": "File uploaded successfully",
                    "file": {
                        "id": db_record.id if db_record else None,
                        "name": file_info["safe_name"],
                        "original_name": file_obj.name,
                        "url": file_info["web_path"],
                        "size": file_obj.size,
                        "type": file_info["extension"],
                        "uploaded_at": timezone.now().isoformat()
                    },
                    "metadata": {
                        "upload_id": str(db_record.id) if db_record else "temp",
                        "checksum": save_result.get("checksum", ""),
                        "storage_path": file_info["full_path"]
                    }
                },
                status=status.HTTP_201_CREATED,
            )

        except Exception as e:
            logger.error(f"❌ File upload failed: {e}", exc_info=True)
            return Response(
                {
                    "success": False,
                    "error": "File upload failed due to server error",
                    "code": "SERVER_ERROR",
                    "details": str(e) if settings.DEBUG else None
                },
                status=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )

    def validate_file(self, file_obj):
        """Comprehensive file validation"""
        # File size validation (max 25MB)
        max_size = 25 * 1024 * 1024
        if file_obj.size > max_size:
            return {
                "valid": False,
                "error": f"File size too large. Maximum size is {max_size // (1024*1024)}MB.",
                "code": "FILE_TOO_LARGE"
            }

        # File extension validation
        allowed_extensions = {
            'images': ['.jpg', '.jpeg', '.png', '.gif', '.bmp', '.webp'],
            'documents': ['.pdf', '.doc', '.docx', '.txt', '.rtf'],
            'archives': ['.zip', '.rar', '.7z'],
            'data': ['.csv', '.xlsx', '.xls', '.json']
        }
        
        file_extension = os.path.splitext(file_obj.name)[1].lower()
        allowed_all = [ext for category in allowed_extensions.values() for ext in category]
        
        if file_extension not in allowed_all:
            return {
                "valid": False,
                "error": f"File type '{file_extension}' not allowed.",
                "code": "INVALID_FILE_TYPE",
                "allowed_types": allowed_extensions
            }

        # MIME type validation (basic)
        suspicious_extensions = ['.exe', '.bat', '.cmd', '.sh', '.php', '.js']
        if file_extension in suspicious_extensions:
            return {
                "valid": False,
                "error": "Potentially dangerous file type detected.",
                "code": "DANGEROUS_FILE_TYPE"
            }

        return {"valid": True}

    def generate_safe_filepath(self, original_filename):
        """Generate safe filepath with organization"""
        # Create organized directory structure
        date_folder = timezone.now().strftime("%Y/%m/%d")
        upload_dir = os.path.join(settings.MEDIA_ROOT, "uploads", date_folder)
        os.makedirs(upload_dir, exist_ok=True)

        # Generate safe filename
        name, ext = os.path.splitext(original_filename)
        safe_name = f"{timezone.now().strftime('%H%M%S')}_{name[:50]}{ext}".replace(" ", "_")
        safe_name = "".join(c for c in safe_name if c.isalnum() or c in "._-")
        
        full_path = os.path.join(upload_dir, safe_name)
        web_path = f"/media/uploads/{date_folder}/{safe_name}"

        return {
            "safe_name": safe_name,
            "full_path": full_path,
            "web_path": web_path,
            "extension": ext.lower()
        }

    def save_file_safely(self, file_obj, file_path):
        """Save file with error handling and checksum"""
        try:
            import hashlib
            file_hash = hashlib.md5()
            
            with open(file_path, "wb+") as destination:
                for chunk in file_obj.chunks():
                    destination.write(chunk)
                    file_hash.update(chunk)
            
            return {
                "success": True,
                "checksum": file_hash.hexdigest(),
                "saved_path": file_path
            }
        except Exception as e:
            # Clean up partially written file
            if os.path.exists(file_path):
                try:
                    os.remove(file_path)
                except:
                    pass
            return {"success": False, "error": str(e)}

    def create_file_record(self, file_obj, file_info, user):
        """Create database record for the uploaded file"""
        try:
            from .models import FileUpload
            
            # Determine file category
            extension = file_info["extension"]
            if extension in ['.jpg', '.jpeg', '.png', '.gif', '.bmp', '.webp']:
                category = 'image'
            elif extension in ['.pdf', '.doc', '.docx', '.txt', '.rtf']:
                category = 'document'
            elif extension in ['.zip', '.rar', '.7z']:
                category = 'archive'
            elif extension in ['.csv', '.xlsx', '.xls', '.json']:
                category = 'data'
            else:
                category = 'other'

            file_record = FileUpload.objects.create(
                file=file_info["web_path"].replace("/media/", ""),
                uploaded_by=user if user.is_authenticated else None,
                title=file_obj.name,
                description=f"Uploaded {timezone.now().strftime('%Y-%m-%d %H:%M')}",
                category=category,
                size_kb=file_obj.size // 1024,
                file_type=extension.replace('.', '').upper(),
                is_public=False  # Default to private
            )
            
            return file_record
        except Exception as e:
            logger.warning(f"⚠️ Could not create file record: {e}")
            return None


# ---------------------------------------------------------------------
# 🔹 ENHANCED HEALTH CHECK ENDPOINT
# ---------------------------------------------------------------------
@api_view(["GET"])
@permission_classes([AllowAny])
def enhanced_health_check(request):
    """Comprehensive health check endpoint with system status"""
    from django.db import connection
    from django.core.cache import cache
    
    health_status = {
        "status": "healthy",
        "timestamp": timezone.now().isoformat(),
        "service": "SeovoSolutions API",
        "version": getattr(settings, "API_VERSION", "2.0.0"),
        "environment": "development" if settings.DEBUG else "production",
        "checks": {}
    }

    # Database check
    try:
        with connection.cursor() as cursor:
            cursor.execute("SELECT 1")
        health_status["checks"]["database"] = {
            "status": "healthy",
            "response_time": "ok"
        }
    except Exception as e:
        health_status["checks"]["database"] = {
            "status": "unhealthy", 
            "error": str(e)
        }
        health_status["status"] = "degraded"

    # Cache check
    try:
        cache_key = "health_check_" + str(timezone.now().timestamp())
        cache.set(cache_key, "ok", 10)
        if cache.get(cache_key) == "ok":
            health_status["checks"]["cache"] = {
                "status": "healthy",
                "backend": settings.CACHES['default']['BACKEND']
            }
        else:
            health_status["checks"]["cache"] = {"status": "unhealthy"}
            health_status["status"] = "degraded"
    except Exception as e:
        health_status["checks"]["cache"] = {
            "status": "unhealthy",
            "error": str(e)
        }
        health_status["status"] = "degraded"

    # Storage check
    try:
        import tempfile
        with tempfile.NamedTemporaryFile(delete=True) as f:
            f.write(b"test")
        health_status["checks"]["storage"] = {"status": "healthy"}
    except Exception as e:
        health_status["checks"]["storage"] = {
            "status": "unhealthy",
            "error": str(e)
        }
        health_status["status"] = "degraded"

    # System metrics
    try:
        from .models import ConversationLog, TrainingRegistration
        health_status["system_metrics"] = {
            "total_conversations": ConversationLog.objects.count(),
            "total_registrations": TrainingRegistration.objects.count(),
            "uptime": "active"  # This could be enhanced with actual uptime tracking
        }
    except Exception as e:
        health_status["system_metrics"] = {"error": "Could not fetch metrics"}

    return Response(health_status)


# ---------------------------------------------------------------------
# 🔹 ENHANCED API ROOT (Comprehensive Overview)
# ---------------------------------------------------------------------
@api_view(["GET"])
@permission_classes([AllowAny])
def enhanced_api_root(request, format=None):
    """Enhanced API root with comprehensive endpoint listing and documentation"""
    base_url = request.build_absolute_uri("/").rstrip("/")
    api_base = f"{base_url}/api"
    
    endpoints = {
        "contacts": {
            "url": f"{api_base}/contacts/",
            "description": "Contact form submissions and inquiries",
            "methods": ["GET", "POST"],
            "authentication": "Optional for POST"
        },
        "training": {
            "url": f"{api_base}/training/",
            "description": "Training program registrations and management",
            "methods": ["GET", "POST", "PUT", "PATCH", "DELETE"],
            "authentication": "Required for admin operations"
        },
        "training-programs": {
            "url": f"{api_base}/training-programs/",
            "description": "Available training programs and courses",
            "methods": ["GET"],
            "authentication": "None"
        },
        "events": {
            "url": f"{api_base}/events/",
            "description": "Events, workshops, and webinars",
            "methods": ["GET", "POST"],
            "authentication": "Required for POST"
        },
        "testimonials": {
            "url": f"{api_base}/testimonials/",
            "description": "Customer testimonials and reviews",
            "methods": ["GET", "POST"],
            "authentication": "Optional for POST"
        },
        "posts": {
            "url": f"{api_base}/posts/",
            "description": "Blog posts and articles",
            "methods": ["GET", "POST"],
            "authentication": "Required for POST"
        },
        "blovion": {
            "url": f"{api_base}/blovion/",
            "description": "Blovion platform posts and content",
            "methods": ["GET", "POST"],
            "authentication": "Required for POST"
        },
        "ai-chat": {
            "url": f"{api_base}/ai-chat/",
            "description": "AI assistant chat interface",
            "methods": ["POST"],
            "authentication": "None"
        },
        "assistant-feedback": {
            "url": f"{api_base}/assistant-feedback/",
            "description": "Feedback for AI assistant responses",
            "methods": ["POST", "GET"],
            "authentication": "None for POST, Admin for GET"
        },
        "file-upload": {
            "url": f"{api_base}/file-upload/",
            "description": "Secure file upload service",
            "methods": ["POST"],
            "authentication": "Optional"
        },
        "stats": {
            "url": f"{api_base}/stats/",
            "description": "Website and business statistics",
            "methods": ["GET"],
            "authentication": "None"
        },
        "health": {
            "url": f"{api_base}/health/",
            "description": "System health and status check",
            "methods": ["GET"],
            "authentication": "None"
        }
    }
    
    # Service information
    service_info = {
        "name": "SeovoSolutions API",
        "version": "2.0.0",
        "description": "Comprehensive backend API for SeovoSolutions training platform",
        "status": "operational",
        "timestamp": timezone.now().isoformat(),
        "support": {
            "email": "support@seovosolutions.com",
            "whatsapp": "+254748370734",
            "documentation": f"{api_base}/docs/"
        },
        "features": {
            "ai_services": ["Gemini", "OpenAI", "DeepSeek"],
            "authentication": ["Session", "Token"],
            "file_handling": True,
            "real_time_analytics": True,
            "background_processing": True,
            "frontend_integration": True,
            "enhanced_models": True
        },
        "model_integration": {
            "frontend_serialization": "to_frontend_dict() methods available",
            "enhanced_validation": "Comprehensive data validation",
            "file_handling": "Integrated file upload system",
            "api_ready": "All models optimized for frontend consumption"
        }
    }
    
    return Response({
        "service": service_info,
        "endpoints": endpoints,
        "quick_start": {
            "register_training": f"POST {api_base}/training/",
            "submit_contact": f"POST {api_base}/contacts/",
            "get_events": f"GET {api_base}/events/",
            "get_posts": f"GET {api_base}/posts/",
            "ai_chat": f"POST {api_base}/ai-chat/",
            "get_stats": f"GET {api_base}/stats/",
            "health_check": f"GET {api_base}/health/"
        },
        "frontend_integration": {
            "data_format": "JSON with ISO dates",
            "image_urls": "Absolute URLs provided",
            "error_handling": "Structured error responses",
            "validation": "Comprehensive frontend validation"
        }
    }, status=status.HTTP_200_OK)


# ---------------------------------------------------------------------
# 🔹 ENHANCED ROUTER SETUP (Comprehensive REST Endpoints)
# ---------------------------------------------------------------------
class EnhancedDefaultRouter(DefaultRouter):
    """Enhanced router with better endpoint organization"""
    
    def get_api_root_view(self, api_urls=None):
        """Return enhanced API root view"""
        return enhanced_api_root
    
    def get_lookup_regex(self, viewset, lookup_prefix=''):
        """Enhanced lookup regex for better URL patterns"""
        # Allow UUIDs and slugs in URLs
        base_regex = super().get_lookup_regex(viewset, lookup_prefix)
        return base_regex

# Initialize enhanced router
router = EnhancedDefaultRouter()

# Core application endpoints with enhanced configurations
if VIEWS_IMPORT_SUCCESS:
    router.register(r"contacts", ContactViewSet, basename="contacts")
    router.register(r"training", EnhancedTrainingViewSet, basename="training")
    router.register(r"training-programs", TrainingProgramViewSet, basename="training-programs")
    router.register(r"events", EventViewSet, basename="events")
    router.register(r"testimonials", TestimonialViewSet, basename="testimonials")
    router.register(r"posts", PostViewSet, basename="posts")
    router.register(r"notifications", NotificationViewSet, basename="notifications")
    router.register(r"blovion", BlovionPostViewSet, basename="blovion")
    router.register(r"file-uploads", FileUploadViewSet, basename="file-uploads")
else:
    logger.warning("⚠️ Using fallback router configuration due to import errors")

# ---------------------------------------------------------------------
# 🔹 ADDITIONAL SERVICE ENDPOINTS
# ---------------------------------------------------------------------

@api_view(['GET'])
@permission_classes([AllowAny])
def service_status(request):
    """Comprehensive service status and feature flags"""
    try:
        # Try to import AI manager safely
        try:
            from .views import ai_manager
            ai_status = ai_manager.get_service_stats() if hasattr(ai_manager, 'get_service_stats') else {"error": "AI manager not available"}
        except ImportError:
            ai_status = {"error": "AI services not available"}
        
        # Check model availability and functionality
        model_status = {}
        try:
            from .models import (
                Contact, TrainingProgram, Event, Post, BlovionPost, 
                Testimonial, FileUpload
            )
            
            # Test model functionality
            test_models = [
                ('Contact', Contact),
                ('TrainingProgram', TrainingProgram),
                ('Event', Event),
                ('Post', Post),
                ('BlovionPost', BlovionPost),
                ('Testimonial', Testimonial),
                ('FileUpload', FileUpload)
            ]
            
            for name, model in test_models:
                try:
                    count = model.objects.count()
                    # Test frontend serialization if object exists
                    if count > 0:
                        instance = model.objects.first()
                        if hasattr(instance, 'to_frontend_dict'):
                            instance.to_frontend_dict()
                    model_status[name] = {
                        "status": "healthy",
                        "count": count,
                        "frontend_serialization": hasattr(model, 'to_frontend_dict')
                    }
                except Exception as e:
                    model_status[name] = {
                        "status": "degraded",
                        "error": str(e),
                        "frontend_serialization": False
                    }
                    
        except ImportError as e:
            model_status = {"error": f"Models not available: {e}"}
        
        status_info = {
            "timestamp": timezone.now().isoformat(),
            "services": {
                "database": "operational",
                "cache": "operational",
                "storage": "operational",
                "ai_services": ai_status,
                "models": model_status
            },
            "feature_flags": {
                "ai_chat": True,
                "file_upload": True,
                "training_registration": True,
                "real_time_analytics": True,
                "email_notifications": getattr(settings, "EMAIL_ENABLED", False),
                "background_tasks": True,
                "frontend_integration": True,
                "enhanced_models": len(model_status) > 0
            },
            "limits": {
                "file_upload_size_mb": 25,
                "ai_request_timeout_seconds": 30,
                "max_concurrent_uploads": 5,
                "rate_limiting": True
            },
            "model_capabilities": {
                "frontend_serialization": any(
                    status.get('frontend_serialization', False) 
                    for status in model_status.values() 
                    if isinstance(status, dict)
                ),
                "total_models_tested": len(model_status),
                "healthy_models": sum(
                    1 for status in model_status.values() 
                    if isinstance(status, dict) and status.get('status') == 'healthy'
                )
            }
        }
        
        return Response(status_info)
    except Exception as e:
        logger.error(f"❌ Service status error: {e}")
        return Response({"error": "Could not determine service status"}, status=500)

@api_view(['POST'])
@permission_classes([IsAuthenticated])
def bulk_operations(request):
    """Enhanced bulk operations endpoint for admin tasks"""
    if not request.user.is_staff:
        return Response({"error": "Admin access required"}, status=403)
    
    try:
        operation = request.data.get('operation')
        data = request.data.get('data', [])
        
        if operation == 'export_training_data':
            from .models import TrainingRegistration
            
            registrations = TrainingRegistration.objects.all()
            serialized_data = [reg.to_frontend_dict() for reg in registrations]
            
            return Response({
                "operation": operation,
                "count": len(serialized_data),
                "data": serialized_data,
                "exported_at": timezone.now().isoformat()
            })
        
        elif operation == 'export_events_data':
            from .models import Event
            
            events = Event.objects.all()
            serialized_data = [event.to_frontend_dict() for event in events]
            
            return Response({
                "operation": operation,
                "count": len(serialized_data),
                "data": serialized_data,
                "exported_at": timezone.now().isoformat()
            })
        
        elif operation == 'export_posts_data':
            from .models import Post
            
            posts = Post.objects.all()
            serialized_data = [post.to_frontend_dict() for post in posts]
            
            return Response({
                "operation": operation,
                "count": len(serialized_data),
                "data": serialized_data,
                "exported_at": timezone.now().isoformat()
            })
        
        elif operation == 'cleanup_old_data':
            # Example cleanup operation
            from .models import ConversationLog
            
            cutoff_date = timezone.now() - timedelta(days=90)
            deleted_count, _ = ConversationLog.objects.filter(
                created_at__lt=cutoff_date
            ).delete()
            
            return Response({
                "operation": operation,
                "deleted_records": deleted_count,
                "cutoff_date": cutoff_date.isoformat()
            })
        
        else:
            return Response({"error": f"Unknown operation: {operation}"}, status=400)
            
    except Exception as e:
        logger.error(f"❌ Bulk operation failed: {e}")
        return Response({"error": str(e)}, status=500)


# ---------------------------------------------------------------------
# 🔹 ENHANCED MODEL TESTING ENDPOINTS
# ---------------------------------------------------------------------
@api_view(['GET'])
@permission_classes([AllowAny])
def test_model_serialization(request):
    """Test endpoint to verify model serialization is working"""
    try:
        from .models import Post, Event, BlovionPost, TrainingProgram
        
        test_results = {}
        
        # Test Post model
        try:
            post = Post.objects.first()
            if post:
                test_results['Post'] = post.to_frontend_dict()
            else:
                test_results['Post'] = {"status": "no_data", "message": "No posts available"}
        except Exception as e:
            test_results['Post'] = {"status": "error", "error": str(e)}
        
        # Test Event model
        try:
            event = Event.objects.first()
            if event:
                test_results['Event'] = event.to_frontend_dict()
            else:
                test_results['Event'] = {"status": "no_data", "message": "No events available"}
        except Exception as e:
            test_results['Event'] = {"status": "error", "error": str(e)}
        
        # Test BlovionPost model
        try:
            blovion_post = BlovionPost.objects.first()
            if blovion_post:
                test_results['BlovionPost'] = blovion_post.to_frontend_dict()
            else:
                test_results['BlovionPost'] = {"status": "no_data", "message": "No blovion posts available"}
        except Exception as e:
            test_results['BlovionPost'] = {"status": "error", "error": str(e)}
        
        # Test TrainingProgram model
        try:
            training_program = TrainingProgram.objects.first()
            if training_program:
                test_results['TrainingProgram'] = training_program.to_frontend_dict()
            else:
                test_results['TrainingProgram'] = {"status": "no_data", "message": "No training programs available"}
        except Exception as e:
            test_results['TrainingProgram'] = {"status": "error", "error": str(e)}
        
        return Response({
            "timestamp": timezone.now().isoformat(),
            "model_serialization_test": test_results,
            "summary": {
                "models_tested": len(test_results),
                "successful": sum(1 for result in test_results.values() if 'status' not in result or result.get('status') != 'error'),
                "errors": sum(1 for result in test_results.values() if result.get('status') == 'error')
            }
        })
        
    except Exception as e:
        return Response({
            "error": f"Model serialization test failed: {str(e)}",
            "timestamp": timezone.now().isoformat()
        }, status=500)


# ---------------------------------------------------------------------
# 🔹 COMPREHENSIVE URL PATTERNS (Enhanced Structure)
# ---------------------------------------------------------------------

# Create v1 patterns list that we can extend conditionally
v1_patterns = [
    path("", enhanced_api_root, name="api-v1-root"),
    path("health/", enhanced_health_check, name="health-check-v1"),
    path("status/", service_status, name="service-status-v1"),
    path("test-models/", test_model_serialization, name="test-models-v1"),
    path("", include(router.urls)),  # Include all router endpoints under v1
]

# Add conditional endpoints to v1 patterns based on import success
if VIEWS_IMPORT_SUCCESS:
    v1_patterns.extend([
        path("ai-chat/", EnhancedAIChatView.as_view(), name="ai-chat-v1"),
        path("assistant-feedback/", EnhancedAssistantFeedbackView.as_view(), name="assistant-feedback-v1"),
    ])
else:
    v1_patterns.extend([
        path("ai-chat/", FallbackView.as_view(), name="ai-chat-v1"),
        path("assistant-feedback/", FallbackView.as_view(), name="assistant-feedback-v1"),
    ])

# Create v2 patterns
v2_patterns = [
    path("", enhanced_api_root, name="api-v2-root"),
    path("health/", enhanced_health_check, name="health-check-v2"),
    path("test-models/", test_model_serialization, name="test-models-v2"),
]

# Main URL patterns
urlpatterns = [
    # Root and core service endpoints
    path("", enhanced_api_root, name="api-root"),
    path("health/", enhanced_health_check, name="health-check"),
    path("status/", service_status, name="service-status"),
    path("test-models/", test_model_serialization, name="test-models"),
    
    # Router includes (all ViewSet endpoints)
    path("", include(router.urls)),
    
    # Enhanced standalone API endpoints
    path("file-upload/", EnhancedFileUploadView.as_view(), name="file-upload"),
    
    # API versioning with enhanced structure
    path("v1/", include(v1_patterns)),
    path("v2/", include(v2_patterns)),
]

# Add conditional endpoints based on import success
if VIEWS_IMPORT_SUCCESS:
    urlpatterns += [
        path("ai-chat/", EnhancedAIChatView.as_view(), name="ai-chat"),
        path("assistant-feedback/", EnhancedAssistantFeedbackView.as_view(), name="assistant-feedback"),
        path("stats/", EnhancedStatsView.as_view(), name="stats"),
        path("health/", HealthCheckView.as_view(), name="health"),
    ]
else:
    # Add fallback endpoints
    urlpatterns += [
        path("ai-chat/", FallbackView.as_view(), name="ai-chat"),
        path("assistant-feedback/", FallbackView.as_view(), name="assistant-feedback"),
        path("stats/", FallbackView.as_view(), name="stats"),
        path("health/", FallbackView.as_view(), name="health"),
    ]

# Common endpoints for all cases
urlpatterns += [
    # Admin and bulk operations
    path("bulk-operations/", bulk_operations, name="bulk-operations"),
]

# ---------------------------------------------------------------------
# 🔹 MEDIA FILE SERVING (Development only)
# ---------------------------------------------------------------------
if settings.DEBUG:
    urlpatterns += [
        re_path(r'^media/(?P<path>.*)$', serve, {
            'document_root': settings.MEDIA_ROOT,
        }),
    ]

# ---------------------------------------------------------------------
# 🔹 ENHANCED BACKGROUND SYNC (Improved with Model Integration)
# ---------------------------------------------------------------------
def schedule_enhanced_data_refresh():
    """
    Enhanced background data sync with comprehensive task management.
    """
    try:
        logger.info("🔄 Scheduling enhanced background data refresh...")
        
        # Try Celery first if available
        try:
            from website.tasks import sync_enhanced_frontend_data
            from ovosolutions_backend import celery_app

            # Schedule multiple background tasks
            tasks = [
                ("website.tasks.sync_enhanced_frontend_data", 30),
                ("website.tasks.cleanup_old_conversations", 3600),  # 1 hour
                ("website.tasks.update_ai_models", 86400),  # 24 hours
            ]
            
            for task_name, delay in tasks:
                eta = timezone.now() + timedelta(seconds=delay)
                celery_app.send_task(task_name, eta=eta)
                
            logger.info(f"✅ {len(tasks)} Celery background tasks scheduled successfully.")
            
        except ImportError:
            logger.debug("Celery not configured, using enhanced threading fallback.")
            
            # Enhanced threading fallback
            import threading
            
            def enhanced_background_sync():
                try:
                    # Try to import sync functions safely
                    try:
                        from website.utils import sync_enhanced_frontend_data, cleanup_old_data
                        # Perform comprehensive data sync
                        sync_enhanced_frontend_data()
                        # Cleanup old data
                        cleanup_old_data()
                    except ImportError:
                        logger.warning("⚠️ Background sync utilities not available")
                    
                    logger.info("✅ Enhanced background data sync completed successfully.")
                    
                    # Log system metric
                    try:
                        from .models import SystemMetric
                        SystemMetric.objects.create(
                            metric_type="background_sync_completed",
                            value=1,
                            metadata={
                                "timestamp": timezone.now().isoformat(),
                                "tasks_completed": ["data_sync", "cleanup"]
                            }
                        )
                    except ImportError:
                        pass
                    
                except Exception as e:
                    logger.error(f"❌ Enhanced background sync failed: {e}")
                    # Log error metric
                    try:
                        from .models import SystemMetric
                        SystemMetric.objects.create(
                            metric_type="background_sync_failed",
                            value=0,
                            metadata={"error": str(e), "timestamp": timezone.now().isoformat()}
                        )
                    except ImportError:
                        pass
            
            # Start enhanced background thread with delay
            timer = threading.Timer(45.0, enhanced_background_sync)
            timer.daemon = True
            timer.start()
            logger.info("✅ Enhanced threaded data refresh scheduled successfully.")
            
    except Exception as e:
        logger.error(f"❌ Enhanced auto-refresh scheduling failed: {e}")


# ---------------------------------------------------------------------
# 🔹 ENHANCED STARTUP INITIALIZATION
# ---------------------------------------------------------------------
def initialize_enhanced_urls():
    """Comprehensive URL initialization with system checks"""
    try:
        logger.info("🚀 Initializing enhanced URL configuration...")
        
        # Schedule enhanced background tasks
        schedule_enhanced_data_refresh()
        
        # Log comprehensive endpoint information
        endpoint_categories = {}
        for pattern in urlpatterns:
            if hasattr(pattern, 'url_patterns'):
                # Include patterns
                for subpattern in pattern.url_patterns:
                    endpoint_categories.setdefault('included', []).append(str(subpattern.pattern))
            else:
                # Direct patterns
                endpoint_categories.setdefault('direct', []).append(str(pattern.pattern))
        
        # Log initialization summary
        total_endpoints = sum(len(endpoints) for endpoints in endpoint_categories.values())
        logger.info(f"✅ Enhanced URL configuration initialized with {total_endpoints} endpoint patterns")
        logger.info(f"📊 Endpoint breakdown: { {k: len(v) for k, v in endpoint_categories.items()} }")
        
        # Log available API versions
        api_versions = [p.pattern for p in urlpatterns if 'v' in str(p.pattern)]
        logger.info(f"🔢 Available API versions: {api_versions}")
        
        # Test model integration
        try:
            from .models import Post, Event, BlovionPost
            model_test_results = {
                'Post': Post.objects.count(),
                'Event': Event.objects.count(), 
                'BlovionPost': BlovionPost.objects.count()
            }
            logger.info(f"📈 Model data counts: {model_test_results}")
        except Exception as e:
            logger.warning(f"⚠️ Model integration test failed: {e}")
        
        # Initialize system metrics for URL startup
        try:
            from .models import SystemMetric
            SystemMetric.objects.create(
                metric_type="url_configuration_initialized",
                value=total_endpoints,
                metadata={
                    "endpoint_categories": {k: len(v) for k, v in endpoint_categories.items()},
                    "api_versions": api_versions,
                    "timestamp": timezone.now().isoformat(),
                    "model_integration": "enhanced"
                }
            )
        except ImportError:
            pass
        
    except Exception as e:
        logger.error(f"❌ Enhanced URL initialization failed: {e}")
        # Log initialization failure
        try:
            from .models import SystemMetric
            SystemMetric.objects.create(
                metric_type="url_initialization_failed",
                value=0,
                metadata={"error": str(e), "timestamp": timezone.now().isoformat()}
            )
        except ImportError:
            pass


# Run enhanced initialization
try:
    initialize_enhanced_urls()
except Exception as e:
    logger.error(f"❌ Enhanced URL initialization failed: {e}")


# ---------------------------------------------------------------------
# 🔹 ENHANCED 404 HANDLER (Comprehensive)
# ---------------------------------------------------------------------
def enhanced_api_404_handler(request, exception=None):
    """
    Enhanced JSON error response for unknown endpoints with intelligent suggestions.
    """
    path = request.path
    
    # Comprehensive endpoint suggestions
    endpoint_suggestions = {
        '/api/contacts/': {
            'description': 'Contact form submissions',
            'methods': ['POST'],
            'example': {'name': 'John', 'email': 'john@example.com', 'message': 'Hello'}
        },
        '/api/training/': {
            'description': 'Training program registrations', 
            'methods': ['POST'],
            'example': {'full_name': 'John Doe', 'email': 'john@example.com', 'program': 'Web Development'}
        },
        '/api/training-programs/': {
            'description': 'Available training programs',
            'methods': ['GET']
        },
        '/api/events/': {
            'description': 'Events and workshops',
            'methods': ['GET']
        },
        '/api/posts/': {
            'description': 'Blog posts and articles', 
            'methods': ['GET']
        },
        '/api/blovion/': {
            'description': 'Blovion data insights',
            'methods': ['GET']
        },
        '/api/ai-chat/': {
            'description': 'AI assistant chat',
            'methods': ['POST'],
            'example': {'message': 'Tell me about web development program'}
        },
        '/api/stats/': {
            'description': 'Website statistics',
            'methods': ['GET']
        },
        '/api/health/': {
            'description': 'System health check',
            'methods': ['GET']
        }
    }
    
    # Find closest matching endpoint
    suggestion = None
    suggested_endpoint = None
    
    for known_path, info in endpoint_suggestions.items():
        if known_path in path or any(segment in path for segment in known_path.strip('/').split('/')):
            suggestion = info
            suggested_endpoint = known_path
            break
    
    # Prepare comprehensive error response
    error_response = {
        "error": {
            "code": "ENDPOINT_NOT_FOUND",
            "message": "The requested endpoint was not found",
            "path": path,
            "method": request.method
        },
        "suggestions": {
            "closest_endpoint": suggested_endpoint,
            "description": suggestion["description"] if suggestion else "Browse available endpoints",
            "available_methods": suggestion["methods"] if suggestion else ["GET", "POST"]
        },
        "documentation": {
            "api_root": "/api/",
            "health_check": "/api/health/",
            "available_endpoints": list(endpoint_suggestions.keys())
        },
        "timestamp": timezone.now().isoformat(),
        "request_id": getattr(request, 'id', 'unknown')
    }
    
    # Add example if available
    if suggestion and 'example' in suggestion:
        error_response["suggestions"]["example_request"] = suggestion["example"]
    
    return JsonResponse(error_response, status=404)


# ---------------------------------------------------------------------
# 🔹 ENHANCED 500 ERROR HANDLER
# ---------------------------------------------------------------------
def enhanced_api_500_handler(request):
    """Comprehensive JSON error response for server errors"""
    error_response = {
        "error": {
            "code": "INTERNAL_SERVER_ERROR",
            "message": "An internal server error occurred",
            "path": request.path,
            "method": request.method
        },
        "support": {
            "email": "support@seovosolutions.com",
            "whatsapp": "+254748370734",
            "status_page": "/api/health/"
        },
        "actions": {
            "retry": "Please try again in a few moments",
            "contact_support": "If the problem persists, contact our support team",
            "check_status": "Check system status at /api/health/"
        },
        "timestamp": timezone.now().isoformat(),
        "request_id": getattr(request, 'id', 'unknown'),
        "correlation_id": f"err_{int(timezone.now().timestamp())}"
    }
    
    # Add debug information in development
    if settings.DEBUG:
        import traceback
        error_response["debug"] = {
            "traceback": traceback.format_exc(),
            "python_version": os.sys.version,
            "django_version": settings.VERSION
        }
    
    return JsonResponse(error_response, status=500)


# ---------------------------------------------------------------------
# 🔹 ENHANCED 400 ERROR HANDLER
# ---------------------------------------------------------------------
def enhanced_api_400_handler(request, exception=None):
    """Enhanced JSON error response for bad requests"""
    error_response = {
        "error": {
            "code": "BAD_REQUEST",
            "message": "The request could not be understood or was missing required parameters",
            "path": request.path,
            "method": request.method
        },
        "suggestions": {
            "check_parameters": "Verify all required parameters are included",
            "validate_data_types": "Ensure data types match expected formats",
            "review_documentation": "Check API documentation for endpoint requirements"
        },
        "timestamp": timezone.now().isoformat(),
        "request_id": getattr(request, 'id', 'unknown')
    }
    
    return JsonResponse(error_response, status=400)


# Register enhanced error handlers
handler404 = enhanced_api_404_handler
handler500 = enhanced_api_500_handler
handler400 = enhanced_api_400_handler


# ---------------------------------------------------------------------
# 🔹 ENHANCED DEBUG URL PATTERNS (Development only)
# ---------------------------------------------------------------------
if settings.DEBUG:
    # Enhanced debug endpoints for development
    @api_view(["GET"])
    @permission_classes([AllowAny])
    def enhanced_debug_endpoints(request):
        """Comprehensive debug endpoint to list all available URLs"""
        from django.urls import get_resolver
        resolver = get_resolver()
        patterns = []
        
        def list_enhanced_patterns(patterns, base=''):
            for pattern in patterns:
                if hasattr(pattern, 'url_patterns'):
                    # Include pattern
                    list_enhanced_patterns(pattern.url_patterns, base + str(pattern.pattern))
                else:
                    # Regular pattern
                    full_pattern = base + str(pattern.pattern)
                    pattern_info = {
                        'pattern': full_pattern,
                        'name': pattern.name or 'No name',
                        'callback': str(pattern.callback) if hasattr(pattern, 'callback') else 'Unknown',
                        'methods': getattr(pattern, 'methods', None) or ['GET']  # Default to GET
                    }
                    
                    # Add additional info for API views
                    if hasattr(pattern.callback, 'cls'):
                        pattern_info['view_class'] = pattern.callback.cls.__name__
                        pattern_info['permission_classes'] = [
                            perm.__name__ for perm in 
                            getattr(pattern.callback.cls, 'permission_classes', [])
                        ]
                    
                    patterns.append(pattern_info)
        
        list_enhanced_patterns(resolver.url_patterns)
        
        # Organize by category
        organized_patterns = {
            'api_endpoints': [p for p in patterns if p['pattern'].startswith('/api')],
            'admin_endpoints': [p for p in patterns if 'admin' in p['pattern']],
            'debug_endpoints': [p for p in patterns if 'debug' in p['pattern']],
            'other_endpoints': [p for p in patterns if not any(
                p['pattern'].startswith(prefix) for prefix in ['/api', '/admin', '/debug']
            )]
        }
        
        return Response({
            "total_endpoints": len(patterns),
            "organized_patterns": organized_patterns,
            "url_configuration": {
                "root_urlconf": settings.ROOT_URLCONF,
                "debug": settings.DEBUG,
                "installed_apps": len(settings.INSTALLED_APPS),
                "model_integration": "enhanced"
            }
        })
    
    @api_view(["GET"])
    @permission_classes([IsAdminUser])
    def system_diagnostics(request):
        """Comprehensive system diagnostics endpoint"""
        import platform
        import psutil
        from django.db import connection
        
        try:
            # System information
            system_info = {
                "python": {
                    "version": platform.python_version(),
                    "implementation": platform.python_implementation(),
                    "compiler": platform.python_compiler()
                },
                "platform": {
                    "system": platform.system(),
                    "release": platform.release(),
                    "version": platform.version(),
                    "machine": platform.machine()
                },
                "django": {
                    "version": settings.VERSION,
                    "debug": settings.DEBUG,
                    "timezone": settings.TIME_ZONE,
                    "allowed_hosts": settings.ALLOWED_HOSTS
                }
            }
            
            # Database information
            db_info = {
                "engine": settings.DATABASES['default']['ENGINE'],
                "name": settings.DATABASES['default']['NAME'],
                "tables": connection.introspection.table_names()
            }
            
            # Process information
            process = psutil.Process()
            process_info = {
                "pid": process.pid,
                "memory_mb": process.memory_info().rss // 1024 // 1024,
                "cpu_percent": process.cpu_percent(),
                "threads": process.num_threads(),
                "connections": len(process.connections())
            }
            
            # Application metrics using enhanced models
            try:
                from .models import (
                    TrainingRegistration, ConversationLog, Contact, 
                    Post, Event, Testimonial, BlovionPost, TrainingProgram
                )
                app_metrics = {
                    "training_registrations": TrainingRegistration.objects.count(),
                    "training_programs": TrainingProgram.objects.count(),
                    "conversation_logs": ConversationLog.objects.count(),
                    "contacts": Contact.objects.count(),
                    "posts": Post.objects.count(),
                    "events": Event.objects.count(),
                    "testimonials": Testimonial.objects.count(),
                    "blovion_posts": BlovionPost.objects.count(),
                }
            except ImportError:
                app_metrics = {"error": "Could not load application metrics"}
            
            return Response({
                "system": system_info,
                "database": db_info,
                "process": process_info,
                "application_metrics": app_metrics,
                "model_integration": {
                    "frontend_serialization": "available",
                    "enhanced_validation": "enabled",
                    "file_handling": "integrated"
                },
                "timestamp": timezone.now().isoformat()
            })
            
        except Exception as e:
            return Response({"error": f"Diagnostics failed: {str(e)}"}, status=500)
    
    # Add enhanced debug URLs only in development
    urlpatterns += [
        path("debug/endpoints/", enhanced_debug_endpoints, name="debug-endpoints"),
        path("debug/diagnostics/", system_diagnostics, name="system-diagnostics"),
    ]


# ---------------------------------------------------------------------
# 🔹 EXPORT URL CONFIGURATION
# ---------------------------------------------------------------------
def get_url_configuration():
    """Utility function to get current URL configuration"""
    return {
        "total_patterns": len(urlpatterns),
        "api_versions": ["v1", "v2"],
        "error_handlers": ["400", "404", "500"],
        "debug_endpoints": settings.DEBUG,
        "media_serving": settings.DEBUG,
        "views_import_success": VIEWS_IMPORT_SUCCESS,
        "model_integration": "enhanced",
        "frontend_ready": True
    }


# Log final configuration
logger.info(f"🎯 Enhanced URL configuration loaded: {get_url_configuration()}")