# website/utils.py
# ---------------------------------------------------
# OvoSolutions Automated Background Data Sync Utility
# ---------------------------------------------------
# Runs every 12 hours in a separate daemon thread.
# Updates key data models and logs sync summaries.
# Safe for both development and production environments.
# ---------------------------------------------------

import threading
import logging
import time
import requests
import os
from datetime import datetime, timedelta
from django.utils import timezone
from django.core.cache import cache
from django.conf import settings

logger = logging.getLogger(__name__)

# Configurable interval for updates (12 hours default)
UPDATE_INTERVAL = int(getattr(settings, 'AUTO_SYNC_INTERVAL', 12 * 60 * 60))  # Configurable
_sync_thread_started = False  # Global flag to prevent duplicates


def fetch_and_update_data():
    """
    Enhanced backend data refresh routine.
    Fetches model counts, ensures backend consistency, and performs data cleanup.
    """
    try:
        from website.models import (
            TrainingRegistration, Event, Testimonial, Post, 
            Notification, Contact, JobApplication, BlovionPost
        )

        logger.info(f"[AutoSync] 🔄 Running backend data refresh at {timezone.now()}")

        # Model statistics collection
        models_to_check = [
            (TrainingRegistration, "Training Registrations"),
            (Event, "Events"),
            (Testimonial, "Testimonials"),
            (Post, "Blog Posts"),
            (Notification, "Notifications"),
            (Contact, "Contact Submissions"),
            (JobApplication, "Job Applications"),
            (BlovionPost, "Blovion Posts"),
        ]
        
        stats = {}
        for model, display_name in models_to_check:
            try:
                count = model.objects.count()
                stats[display_name] = count
                logger.info(f"[AutoSync] 📊 {display_name}: {count} records found.")
                
                # Perform data maintenance for specific models
                if model == Notification:
                    cleanup_old_notifications()
                elif model == Contact:
                    archive_old_contacts()
                    
            except Exception as e:
                logger.error(f"[AutoSync] ❌ Error counting {display_name}: {e}")

        # Cache the statistics for dashboard
        cache.set('model_statistics', stats, 3600)  # Cache for 1 hour

        # Sync with external services if configured
        if getattr(settings, 'ENABLE_EXTERNAL_SYNC', False):
            sync_with_external_services()

        # Update last sync timestamp
        cache.set('last_sync_time', timezone.now().isoformat(), UPDATE_INTERVAL)
        
        logger.info(f"[AutoSync] ✅ Data refresh completed successfully at {datetime.now()}")
        logger.info(f"[AutoSync] 📈 Statistics: {stats}")

    except Exception as e:
        logger.exception(f"[AutoSync] ❌ Error during backend sync: {e}")


def cleanup_old_notifications(days_old=30):
    """Clean up notifications older than specified days"""
    try:
        from website.models import Notification
        cutoff_date = timezone.now() - timedelta(days=days_old)
        old_notifications = Notification.objects.filter(created_at__lt=cutoff_date)
        count = old_notifications.count()
        old_notifications.delete()
        if count > 0:
            logger.info(f"[AutoSync] 🗑️ Cleaned up {count} notifications older than {days_old} days")
    except Exception as e:
        logger.error(f"[AutoSync] ❌ Error cleaning up notifications: {e}")


def archive_old_contacts(days_old=90):
    """Archive old contact submissions (could be extended to move to cold storage)"""
    try:
        from website.models import Contact
        cutoff_date = timezone.now() - timedelta(days=days_old)
        old_contacts = Contact.objects.filter(created_at__lt=cutoff_date, archived=False)
        count = old_contacts.count()
        # Instead of deleting, we mark them as archived
        old_contacts.update(archived=True)
        if count > 0:
            logger.info(f"[AutoSync] 📦 Archived {count} contact submissions older than {days_old} days")
    except Exception as e:
        logger.error(f"[AutoSync] ❌ Error archiving contacts: {e}")


def sync_with_external_services():
    """Sync data with external services if configured"""
    try:
        # Example: Sync with external CRM, analytics, etc.
        external_services = getattr(settings, 'EXTERNAL_SERVICES', [])
        
        for service in external_services:
            try:
                # Placeholder for actual sync logic
                logger.info(f"[AutoSync] 🔗 Syncing with {service}")
                # Add actual sync implementation here
                time.sleep(1)  # Simulate API call
            except Exception as e:
                logger.error(f"[AutoSync] ❌ Error syncing with {service}: {e}")
                
    except Exception as e:
        logger.error(f"[AutoSync] ❌ Error in external services sync: {e}")


def schedule_next_run():
    """
    Schedule the next background refresh after the defined interval.
    Runs in a separate daemon thread, never blocking Django's main loop.
    """
    try:
        timer = threading.Timer(UPDATE_INTERVAL, run_auto_sync)
        timer.daemon = True
        timer.start()
        logger.debug(f"[AutoSync] ⏰ Next sync scheduled in {UPDATE_INTERVAL} seconds")
    except Exception as e:
        logger.error(f"[AutoSync] ❌ Error scheduling next run: {e}")
        # Attempt to restart the scheduler after a delay
        threading.Timer(60, schedule_next_run).start()


def run_auto_sync():
    """
    Executes the sync and re-schedules itself continuously.
    """
    try:
        logger.debug("[AutoSync] 🚀 Starting auto-sync cycle")
        fetch_and_update_data()
    except Exception as e:
        logger.error(f"[AutoSync] ⚠️ Unexpected error in run_auto_sync: {e}")
    finally:
        # Always reschedule even if error occurs
        schedule_next_run()


def start_auto_sync():
    """
    Starts the background scheduler once Django loads.
    Safe to call from AppConfig.ready() — only starts once.
    """
    global _sync_thread_started

    if _sync_thread_started:
        logger.debug("[AutoSync] Scheduler already running — skipping duplicate start.")
        return False

    # Check if auto-sync is enabled
    if not getattr(settings, 'ENABLE_AUTO_SYNC', True):
        logger.info("[AutoSync] Auto-sync is disabled in settings.")
        return False

    _sync_thread_started = True
    logger.info(f"[AutoSync] 🕒 Initializing OvoSolutions auto-sync scheduler (every {UPDATE_INTERVAL//3600}h)...")

    try:
        # Start the sync thread
        thread = threading.Thread(target=run_auto_sync, daemon=True, name="AutoSync-Thread")
        thread.start()

        logger.info("[AutoSync] 🚀 Background auto-sync scheduler started successfully.")
        return True
        
    except Exception as e:
        logger.error(f"[AutoSync] ❌ Failed to start auto-sync scheduler: {e}")
        _sync_thread_started = False
        return False


def stop_auto_sync():
    """
    Stops the auto-sync scheduler (for testing or maintenance).
    Note: This is advisory as daemon threads can't be forcefully stopped.
    """
    global _sync_thread_started
    _sync_thread_started = False
    logger.info("[AutoSync] 🛑 Auto-sync scheduler stopped.")


def get_sync_status():
    """
    Returns the current status of the auto-sync system.
    """
    last_sync = cache.get('last_sync_time')
    stats = cache.get('model_statistics', {})
    
    return {
        'is_running': _sync_thread_started,
        'last_sync': last_sync,
        'next_sync_in_seconds': UPDATE_INTERVAL,
        'statistics': stats,
        'update_interval_hours': UPDATE_INTERVAL // 3600,
    }


def publish_to_social_media(post):
    """
    Enhanced social media publishing function.
    Publishes posts to LinkedIn, Facebook, Instagram, and TikTok if credentials exist.
    """
    logger.info(f"🌍 Auto-posting '{post.title}' to social platforms...")

    # Get social media credentials
    linkedin_token = os.getenv("LINKEDIN_ACCESS_TOKEN")
    facebook_token = os.getenv("FACEBOOK_ACCESS_TOKEN")
    instagram_token = os.getenv("INSTAGRAM_ACCESS_TOKEN")
    tiktok_token = os.getenv("TIKTOK_ACCESS_TOKEN")

    results = {
        'linkedin': {'success': False, 'message': 'Not attempted'},
        'facebook': {'success': False, 'message': 'Not attempted'},
        'instagram': {'success': False, 'message': 'Not attempted'},
        'tiktok': {'success': False, 'message': 'Not attempted'}
    }

    # Prepare post content
    title = getattr(post, 'title', '')
    content = getattr(post, 'content', '') or getattr(post, 'caption', '') or ""
    excerpt = content[:200] + "..." if len(content) > 200 else content

    # LinkedIn
    if linkedin_token:
        try:
            # Note: LinkedIn API requires specific formatting and permissions
            response = requests.post(
                "https://api.linkedin.com/v2/ugcPosts",
                headers={
                    "Authorization": f"Bearer {linkedin_token}",
                    "Content-Type": "application/json"
                },
                json={
                    "author": f"urn:li:organization:{os.getenv('LINKEDIN_ORG_ID')}",
                    "lifecycleState": "PUBLISHED",
                    "specificContent": {
                        "com.linkedin.ugc.ShareContent": {
                            "shareCommentary": {
                                "text": f"{title}\n\n{excerpt}"
                            },
                            "shareMediaCategory": "NONE"
                        }
                    },
                    "visibility": {
                        "com.linkedin.ugc.MemberNetworkVisibility": "PUBLIC"
                    }
                },
                timeout=30
            )
            if response.status_code == 201:
                results['linkedin'] = {'success': True, 'message': 'Posted successfully'}
                logger.info("✅ Posted to LinkedIn")
            else:
                results['linkedin'] = {'success': False, 'message': f"HTTP {response.status_code}: {response.text}"}
                logger.warning(f"⚠️ LinkedIn post failed: {response.status_code}")
                
        except Exception as e:
            results['linkedin'] = {'success': False, 'message': str(e)}
            logger.error(f"❌ Error posting to LinkedIn: {e}")

    # Facebook
    if facebook_token:
        try:
            response = requests.post(
                f"https://graph.facebook.com/v19.0/{os.getenv('FACEBOOK_PAGE_ID')}/feed",
                params={"access_token": facebook_token},
                data={
                    "message": f"{title}\n\n{excerpt}",
                    "link": getattr(post, 'absolute_url', '')
                },
                timeout=30
            )
            if response.status_code == 200:
                results['facebook'] = {'success': True, 'message': 'Posted successfully'}
                logger.info("✅ Posted to Facebook")
            else:
                results['facebook'] = {'success': False, 'message': f"HTTP {response.status_code}: {response.text}"}
                logger.warning(f"⚠️ Facebook post failed: {response.status_code}")
                
        except Exception as e:
            results['facebook'] = {'success': False, 'message': str(e)}
            logger.error(f"❌ Error posting to Facebook: {e}")

    # Instagram (Placeholder - requires more complex setup)
    if instagram_token:
        try:
            # Instagram API requires media upload and publishing in separate steps
            # This is a simplified placeholder
            logger.info("📷 Instagram post prepared (requires manual media upload)")
            results['instagram'] = {'success': True, 'message': 'Post prepared - manual media upload required'}
            
        except Exception as e:
            results['instagram'] = {'success': False, 'message': str(e)}
            logger.error(f"❌ Error preparing Instagram post: {e}")

    # TikTok (Placeholder - requires business account and approval)
    if tiktok_token:
        try:
            # TikTok API access is limited and requires business account
            logger.info("🎵 TikTok post prepared (requires business account access)")
            results['tiktok'] = {'success': True, 'message': 'Post prepared - business account required'}
            
        except Exception as e:
            results['tiktok'] = {'success': False, 'message': str(e)}
            logger.error(f"❌ Error preparing TikTok post: {e}")

    # Log overall results
    successful_posts = sum(1 for platform in results.values() if platform['success'])
    logger.info(f"📊 Social media posting completed: {successful_posts}/4 platforms successful")
    
    return results


def manual_data_sync():
    """
    Manual trigger for data sync - useful for admin actions.
    """
    logger.info("[ManualSync] 🚀 Manual data sync triggered")
    try:
        fetch_and_update_data()
        return True
    except Exception as e:
        logger.error(f"[ManualSync] ❌ Manual sync failed: {e}")
        return False


# Initialize on module import if in development
if settings.DEBUG and getattr(settings, 'START_SYNC_ON_IMPORT', False):
    start_auto_sync()