# website/views.py - PRODUCTION OPTIMIZED
import os
import json
import pickle
import logging
import random
import threading
import requests
import re
import time
from datetime import datetime, timedelta
from typing import Dict, List, Optional, Tuple, Any
from functools import lru_cache

import faiss
import numpy as np
from django.conf import settings
from django.core.cache import cache
from django.core.mail import send_mail, EmailMultiAlternatives
from django.db import models as db_models
from django.db.models import Count, Q, F, Avg, Max, Min
from django.utils import timezone
from django.utils.decorators import method_decorator
from django.views.decorators.cache import cache_page
from django.views.decorators.vary import vary_on_cookie
from django.views.decorators.csrf import csrf_exempt
from rest_framework import generics, status, viewsets, permissions, filters, serializers
from rest_framework.parsers import JSONParser, MultiPartParser, FormParser
from rest_framework.response import Response
from rest_framework.views import APIView
from rest_framework.decorators import action, api_view, parser_classes, permission_classes
from rest_framework.pagination import PageNumberPagination
from rest_framework.throttling import AnonRateThrottle, UserRateThrottle
from sklearn.feature_extraction.text import TfidfVectorizer
from sklearn.linear_model import LogisticRegression

# Configure structured logging for production
logger = logging.getLogger('website')

# =========================================================================
# SAFE IMPORTS WITH PRODUCTION FALLBACKS
# =========================================================================

# Track what's available
FEATURES = {
    'ml_models': False,
    'ai_services': False,
    'database': True,  # Assume database is available
    'cache': True,
    'email': True,
}

# Try imports with comprehensive error handling
try:
    import google.generativeai as genai
    GEMINI_AVAILABLE = True
    FEATURES['ai_services'] = True
except ImportError as e:
    genai = None
    GEMINI_AVAILABLE = False
    logger.warning(f"Google Generative AI not available: {e}")

try:
    from sentence_transformers import SentenceTransformer
    SENTENCE_TRANSFORMERS_AVAILABLE = True
    FEATURES['ml_models'] = True
except ImportError as e:
    SentenceTransformer = None
    SENTENCE_TRANSFORMERS_AVAILABLE = False
    logger.warning(f"Sentence Transformers not available: {e}")

# Try to import models safely
try:
    from .models import (
        Document, ConversationLog, AssistantFeedback, Contact, 
        TrainingRegistration, FileUpload, Post, 
        Event, Testimonial, Notification, BlovionPost, TrainingProgram,
        Category, PageView, SystemMetric, SiteConfig
    )
    from .serializers import (
        ConversationLogSerializer, ContactSerializer, 
        TrainingRegistrationSerializer, FileUploadSerializer, 
        PostSerializer, EventSerializer, TestimonialSerializer, 
        NotificationSerializer, DocumentSerializer, AssistantFeedbackSerializer,
        BlovionPostSerializer, TrainingProgramSerializer, CategorySerializer, 
        PageViewSerializer, SystemMetricSerializer, SiteConfigSerializer
    )
    MODELS_AVAILABLE = True
except Exception as e:
    logger.error(f"Models import failed: {e}")
    MODELS_AVAILABLE = False
    FEATURES['database'] = False
    # Create safe dummy classes
    class BaseModel:
        objects = type('Manager', (), {
            'all': lambda: [],
            'filter': lambda **kwargs: [],
            'get': lambda **kwargs: None,
            'create': lambda **kwargs: type('obj', (), {'id': 1})(),
            'count': lambda: 0,
            'values': lambda *args: [],
            'annotate': lambda **kwargs: [],
        })()
    
    # Create safe dummy models to prevent crashes
    class TrainingRegistration(BaseModel): pass
    class TrainingProgram(BaseModel): pass
    class ConversationLog(BaseModel): pass
    class AssistantFeedback(BaseModel): pass
    class SystemMetric(BaseModel): pass
    class Notification(BaseModel): pass
    class Post(BaseModel): pass
    class BlovionPost(BaseModel): pass
    class Testimonial(BaseModel): pass
    class Contact(BaseModel): pass
    class Event(BaseModel): pass

# =========================================================================
# PRODUCTION CONFIGURATION WITH CACHING
# =========================================================================

class ProductionConfig:
    """Production configuration manager with caching"""
    
    # AI Configuration with caching
    @staticmethod
    @lru_cache(maxsize=1)
    def get_ai_config():
        """Get AI configuration with caching"""
        config = {
            'gemini': {
                'key': getattr(settings, "GEMINI_API_KEY", os.getenv("GEMINI_API_KEY", "")),
                'model': getattr(settings, "GEMINI_MODEL_NAME", "gemini-1.5-flash"),
                'enabled': bool(getattr(settings, "GEMINI_API_KEY", "")) and GEMINI_AVAILABLE,
            },
            'openai': {
                'key': getattr(settings, "OPENAI_API_KEY", os.getenv("OPENAI_API_KEY", "")),
                'model': getattr(settings, "OPENAI_MODEL_NAME", "gpt-3.5-turbo"),
                'enabled': bool(getattr(settings, "OPENAI_API_KEY", "")),
            },
            'deepseek': {
                'key': getattr(settings, "DEEPSEEK_API_KEY", os.getenv("DEEPSEEK_API_KEY", "")),
                'model': getattr(settings, "DEEPSEEK_MODEL_NAME", "deepseek-chat"),
                'enabled': bool(getattr(settings, "DEEPSEEK_API_KEY", "")),
            }
        }
        
        # Configure Gemini if available
        if config['gemini']['enabled'] and GEMINI_AVAILABLE:
            try:
                genai.configure(api_key=config['gemini']['key'])
                logger.info("✅ Gemini API configured")
            except Exception as e:
                logger.error(f"❌ Gemini configuration failed: {e}")
                config['gemini']['enabled'] = False
        
        return config
    
    # Embedding model configuration
    @staticmethod
    @lru_cache(maxsize=1)
    def get_embedding_model():
        """Get or create embedding model with caching"""
        if not SENTENCE_TRANSFORMERS_AVAILABLE:
            return None
        
        model_name = getattr(settings, "EMBED_MODEL_NAME", "all-MiniLM-L6-v2")
        cache_key = f"embedding_model_{model_name}"
        
        cached_model = cache.get(cache_key)
        if cached_model:
            logger.info("✅ Loaded embedding model from cache")
            return cached_model
        
        try:
            model = SentenceTransformer(model_name)
            cache.set(cache_key, model, 3600)  # Cache for 1 hour
            logger.info(f"✅ Created embedding model: {model_name}")
            return model
        except Exception as e:
            logger.error(f"❌ Failed to create embedding model: {e}")
            return None
    
    # Path configurations
    @staticmethod
    def get_paths():
        """Get file paths with fallbacks"""
        base_dir = getattr(settings, "BASE_DIR", ".")
        return {
            'faiss_index': getattr(settings, "FAISS_INDEX_PATH", os.path.join(base_dir, "faiss_index.bin")),
            'doc_json': getattr(settings, "DOCS_JSON_PATH", os.path.join(base_dir, "doc_texts.json")),
            'intent_model': getattr(settings, "INTENT_MODEL_PATH", os.path.join(base_dir, "intent_model.pkl")),
            'ai_data_dir': getattr(settings, "AI_DATA_DIR", os.path.join(base_dir, "ai_data")),
        }

config = ProductionConfig()
AI_CONFIG = config.get_ai_config()
PATHS = config.get_paths()

# =========================================================================
# PRODUCTION SAFETY WRAPPERS
# =========================================================================

class ProductionSafety:
    """Production safety wrappers for all operations"""
    
    @staticmethod
    def database_operation(func):
        """Decorator for safe database operations"""
        def wrapper(*args, **kwargs):
            try:
                if not MODELS_AVAILABLE:
                    logger.warning("⚠️ Database operations disabled - models not available")
                    return None
                return func(*args, **kwargs)
            except Exception as e:
                logger.error(f"❌ Database operation failed: {e}")
                return None
        return wrapper
    
    @staticmethod
    def email_operation(func):
        """Decorator for safe email operations"""
        def wrapper(*args, **kwargs):
            try:
                if not settings.EMAIL_HOST_USER:
                    logger.warning("⚠️ Email configuration missing")
                    return False
                return func(*args, **kwargs)
            except Exception as e:
                logger.error(f"❌ Email operation failed: {e}")
                return False
        return wrapper
    
    @staticmethod
    def external_service_operation(func):
        """Decorator for safe external service calls"""
        def wrapper(*args, **kwargs):
            start_time = time.time()
            try:
                result = func(*args, **kwargs)
                duration = time.time() - start_time
                logger.info(f"✅ External service completed in {duration:.2f}s")
                return result
            except requests.exceptions.Timeout:
                logger.error("❌ External service timeout")
                return None
            except requests.exceptions.ConnectionError:
                logger.error("❌ External service connection error")
                return None
            except Exception as e:
                logger.error(f"❌ External service error: {e}")
                return None
        return wrapper
    
    @staticmethod
    def create_system_metric(metric_type, value=1, metadata=None, tags=None):
        """Safely create system metrics"""
        try:
            if not MODELS_AVAILABLE:
                # Log to cache as fallback
                cache_key = f"metric_fallback_{int(time.time())}"
                cache.set(cache_key, {
                    'type': metric_type,
                    'value': value,
                    'metadata': metadata or {},
                    'timestamp': timezone.now().isoformat(),
                    'tags': tags or []
                }, 86400)  # Cache for 24 hours
                return None
            
            metric = SystemMetric.objects.create(
                metric_type=metric_type,
                value=value,
                metadata=metadata or {},
                recorded_at=timezone.now()
            )
            logger.debug(f"📊 Created system metric: {metric_type}")
            return metric
        except Exception as e:
            logger.error(f"❌ Failed to create system metric: {e}")
            return None

safety = ProductionSafety()

# =========================================================================
# ENHANCED AI SERVICE MANAGER - PRODUCTION READY
# =========================================================================

class ProductionAIServiceManager:
    """Production-ready AI service manager with rate limiting and circuit breaking"""
    
    def __init__(self):
        self.config = AI_CONFIG
        self.stats = {
            'total_requests': 0,
            'successful_requests': 0,
            'failed_requests': 0,
            'circuit_breaker': {}  # Track service health
        }
        self.request_history = []  # Store recent requests for rate limiting
        
    def _rate_limit_check(self, service_name):
        """Check if service is rate limited"""
        now = time.time()
        # Keep only requests from last minute
        self.request_history = [t for t in self.request_history if now - t < 60]
        
        # Service-specific rate limits
        limits = {
            'gemini': 60,  # 60 requests per minute
            'openai': 60,  # 60 requests per minute
            'deepseek': 60,  # 60 requests per minute
        }
        
        limit = limits.get(service_name, 30)
        recent_count = len(self.request_history)
        
        if recent_count >= limit:
            logger.warning(f"⚠️ Rate limit reached for {service_name}: {recent_count}/{limit}")
            return False
        
        self.request_history.append(now)
        return True
    
    def _circuit_breaker_check(self, service_name):
        """Circuit breaker pattern - check if service should be skipped"""
        if service_name not in self.stats['circuit_breaker']:
            return True
        
        breaker = self.stats['circuit_breaker'][service_name]
        if breaker.get('state') == 'open':
            # Check if enough time has passed to try again
            if time.time() - breaker.get('opened_at', 0) > 60:  # 60 seconds
                breaker['state'] = 'half-open'
                breaker['attempts'] = 0
            else:
                return False
        
        return True
    
    def _update_circuit_breaker(self, service_name, success):
        """Update circuit breaker state"""
        if service_name not in self.stats['circuit_breaker']:
            self.stats['circuit_breaker'][service_name] = {
                'state': 'closed',
                'failures': 0,
                'successes': 0,
                'opened_at': 0
            }
        
        breaker = self.stats['circuit_breaker'][service_name]
        
        if success:
            breaker['successes'] += 1
            breaker['failures'] = 0
            if breaker['state'] == 'half-open':
                breaker['state'] = 'closed'
        else:
            breaker['failures'] += 1
            breaker['successes'] = 0
            
            # Open circuit if too many failures
            if breaker['failures'] >= 5:
                breaker['state'] = 'open'
                breaker['opened_at'] = time.time()
                logger.warning(f"⚠️ Circuit breaker opened for {service_name}")
    
    @safety.external_service_operation
    def call_gemini(self, prompt: str) -> Optional[str]:
        """Call Gemini API with production error handling"""
        if not self.config['gemini']['enabled']:
            return None
        
        if not self._rate_limit_check('gemini'):
            return None
        
        if not self._circuit_breaker_check('gemini'):
            return None
        
        try:
            model = genai.GenerativeModel(self.config['gemini']['model'])
            response = model.generate_content(
                prompt,
                generation_config=genai.types.GenerationConfig(
                    temperature=0.7,
                    top_p=0.8,
                    top_k=40,
                    max_output_tokens=1000,
                )
            )
            self._update_circuit_breaker('gemini', True)
            return response.text
        except Exception as e:
            logger.error(f"❌ Gemini API error: {e}")
            self._update_circuit_breaker('gemini', False)
            return None
    
    @safety.external_service_operation
    def call_openai(self, prompt: str) -> Optional[str]:
        """Call OpenAI API with production error handling"""
        if not self.config['openai']['enabled']:
            return None
        
        if not self._rate_limit_check('openai'):
            return None
        
        if not self._circuit_breaker_check('openai'):
            return None
        
        try:
            headers = {
                "Authorization": f"Bearer {self.config['openai']['key']}",
                "Content-Type": "application/json"
            }
            data = {
                "model": self.config['openai']['model'],
                "messages": [{"role": "user", "content": prompt}],
                "max_tokens": 1000,
                "temperature": 0.7
            }
            
            response = requests.post(
                "https://api.openai.com/v1/chat/completions",
                headers=headers,
                json=data,
                timeout=10  # Shorter timeout for production
            )
            
            if response.status_code == 200:
                result = response.json()
                self._update_circuit_breaker('openai', True)
                return result['choices'][0]['message']['content']
            else:
                logger.error(f"❌ OpenAI API error: {response.status_code}")
                self._update_circuit_breaker('openai', False)
                return None
        except Exception as e:
            logger.error(f"❌ OpenAI API exception: {e}")
            self._update_circuit_breaker('openai', False)
            return None
    
    def get_response(self, prompt: str, preferred_service: str = None) -> Tuple[str, str]:
        """Get AI response with intelligent service selection"""
        self.stats['total_requests'] += 1
        
        # Service priority order
        services = ['gemini', 'openai', 'deepseek']
        
        # Use preferred service if specified and available
        if preferred_service and self.config.get(preferred_service, {}).get('enabled', False):
            services.remove(preferred_service)
            services.insert(0, preferred_service)
        
        # Filter to enabled services
        available_services = [s for s in services if self.config.get(s, {}).get('enabled', False)]
        
        if not available_services:
            logger.warning("⚠️ No AI services available")
            return self._get_fallback_response(prompt), "fallback"
        
        # Try each service
        for service_name in available_services:
            logger.info(f"🔄 Trying AI service: {service_name}")
            
            if service_name == 'gemini':
                response = self.call_gemini(prompt)
            elif service_name == 'openai':
                response = self.call_openai(prompt)
            else:
                continue
            
            if response and response.strip():
                self.stats['successful_requests'] += 1
                logger.info(f"✅ Got response from {service_name}")
                return response, service_name
        
        # All services failed
        self.stats['failed_requests'] += 1
        logger.warning("⚠️ All AI services failed")
        return self._get_fallback_response(prompt), "fallback"
    
    def _get_fallback_response(self, user_input: str) -> str:
        """Get intelligent fallback response"""
        cache_key = f"fallback_response_{hash(user_input) % 100}"
        cached = cache.get(cache_key)
        if cached:
            return cached
        
        responses = [
            "I understand you're asking about our services. For the most current information about our training programs, pricing, and schedules, please contact our team directly at +254748370734 or email training@seovosolutions.com.",
            "Great question! Our AI assistant is currently learning. For detailed information about our programs, please reach out to our team who can provide personalized assistance.",
            "Thanks for your inquiry! Our training programs include web development, data science, and personalized mentorship. Contact us for a free consultation to discuss your learning goals.",
            "I'd love to help you with that! For specific program details and personalized guidance, our human team is available to assist you. You can reach them at +254748370734."
        ]
        
        response = random.choice(responses)
        cache.set(cache_key, response, 300)  # Cache for 5 minutes
        return response
    
    def get_stats(self) -> Dict[str, Any]:
        """Get service statistics"""
        return {
            **self.stats,
            'available_services': [s for s, config in self.config.items() if config.get('enabled', False)],
            'circuit_breaker_states': self.stats['circuit_breaker'],
            'success_rate': (self.stats['successful_requests'] / self.stats['total_requests'] * 100 
                           if self.stats['total_requests'] > 0 else 0)
        }

# Initialize AI manager
ai_manager = ProductionAIServiceManager()

# =========================================================================
# ENHANCED VIEWSETS WITH PRODUCTION FEATURES
# =========================================================================

class ProductionViewSet(viewsets.ModelViewSet):
    """Base ViewSet with production features"""
    throttle_classes = [AnonRateThrottle, UserRateThrottle]
    parser_classes = [JSONParser, MultiPartParser, FormParser]
    
    def handle_exception(self, exc):
        """Global exception handler with logging"""
        logger.error(f"❌ ViewSet error in {self.__class__.__name__}: {exc}")
        return super().handle_exception(exc)
    
    def perform_create(self, serializer):
        """Enhanced create with metrics"""
        instance = serializer.save()
        safety.create_system_metric(
            f"{self.__class__.__name__.lower()}_created",
            metadata={'id': instance.id}
        )
        return instance

class TrainingProgramViewSet(ProductionViewSet):
    """Training programs with caching and production optimizations"""
    queryset = TrainingProgram.objects.filter(is_active=True) if MODELS_AVAILABLE else []
    serializer_class = TrainingProgramSerializer
    permission_classes = [permissions.AllowAny]
    filter_backends = [filters.SearchFilter, filters.OrderingFilter]
    search_fields = ['title', 'description', 'program_type', 'level']
    ordering_fields = ['created_at', 'price', 'start_date']
    ordering = ['-created_at']
    
    @method_decorator(cache_page(60 * 5))  # Cache for 5 minutes
    def list(self, request, *args, **kwargs):
        """Cached list view"""
        return super().list(request, *args, **kwargs)
    
    @action(detail=False, methods=['get'])
    def featured(self, request):
        """Get featured programs"""
        cache_key = 'featured_training_programs'
        cached = cache.get(cache_key)
        
        if cached:
            return Response(cached)
        
        programs = self.get_queryset().filter(featured=True)[:6]
        serializer = self.get_serializer(programs, many=True)
        data = serializer.data
        
        cache.set(cache_key, data, 60 * 10)  # Cache for 10 minutes
        return Response(data)
    
    @action(detail=False, methods=['get'])
    def upcoming(self, request):
        """Get upcoming programs"""
        cache_key = 'upcoming_training_programs'
        cached = cache.get(cache_key)
        
        if cached:
            return Response(cached)
        
        now = timezone.now().date()
        programs = self.get_queryset().filter(
            start_date__gte=now,
            registration_open=True
        )[:10]
        
        serializer = self.get_serializer(programs, many=True)
        data = serializer.data
        
        cache.set(cache_key, data, 60 * 15)  # Cache for 15 minutes
        return Response(data)

class EnhancedTrainingRegistrationViewSet(ProductionViewSet):
    """Production-ready training registration"""
    queryset = TrainingRegistration.objects.all() if MODELS_AVAILABLE else []
    serializer_class = TrainingRegistrationSerializer
    permission_classes = [permissions.AllowAny]
    
    def create(self, request, *args, **kwargs):
        """Enhanced create with validation and notifications"""
        # Validate request data
        serializer = self.get_serializer(data=request.data)
        if not serializer.is_valid():
            logger.warning(f"❌ Registration validation failed: {serializer.errors}")
            return Response({
                'error': 'Validation failed',
                'details': serializer.errors
            }, status=status.HTTP_400_BAD_REQUEST)
        
        try:
            # Save registration
            self.perform_create(serializer)
            
            # Send notifications
            self._send_notifications(serializer.instance)
            
            # Return success response
            headers = self.get_success_headers(serializer.data)
            return Response({
                'success': True,
                'message': 'Registration successful!',
                'registration_id': serializer.instance.id,
                'registration_code': serializer.instance.registration_code,
                'next_steps': 'Our team will contact you within 24 hours.'
            }, status=status.HTTP_201_CREATED, headers=headers)
            
        except Exception as e:
            logger.error(f"❌ Registration failed: {e}")
            return Response({
                'error': 'Registration failed',
                'details': str(e)
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
    
    @safety.database_operation
    @safety.email_operation
    def _send_notifications(self, registration):
        """Send registration notifications"""
        # Send admin notification
        try:
            Notification.objects.create(
                type='info',
                title='New Training Registration',
                message=f'New registration from {registration.full_name} for {registration.program}',
                priority='medium',
                payload={
                    'registration_id': registration.id,
                    'email': registration.email,
                    'phone': registration.phone,
                    'program': str(registration.program),
                    'timestamp': timezone.now().isoformat()
                }
            )
        except Exception as e:
            logger.error(f"❌ Failed to create notification: {e}")
        
        # Send confirmation email
        try:
            subject = f'🎉 Registration Confirmation - {registration.program}'
            
            html_content = f"""
            <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
                <h2 style="color: #4F46E5;">Registration Confirmation</h2>
                <p>Dear {registration.full_name},</p>
                <p>Thank you for registering for our <strong>{registration.program}</strong> training program!</p>
                
                <div style="background: #F3F4F6; padding: 20px; border-radius: 8px; margin: 20px 0;">
                    <h3 style="margin-top: 0;">Registration Details:</h3>
                    <p><strong>Registration ID:</strong> {registration.registration_code}</p>
                    <p><strong>Program:</strong> {registration.program}</p>
                    <p><strong>Email:</strong> {registration.email}</p>
                    <p><strong>Phone:</strong> {registration.phone or 'Not provided'}</p>
                    <p><strong>Experience Level:</strong> {registration.experience_level or 'Not specified'}</p>
                </div>
                
                <p><strong>What's Next?</strong></p>
                <ol>
                    <li>Our team will contact you within 24 hours</li>
                    <li>We'll schedule a consultation call</li>
                    <li>We'll create your personalized learning plan</li>
                </ol>
                
                <p>Need immediate assistance? Contact us:</p>
                <ul>
                    <li>📞 WhatsApp: +254748370734</li>
                    <li>📧 Email: training@seovosolutions.com</li>
                </ul>
                
                <p>Best regards,<br>Seovo Solutions Team</p>
            </div>
            """
            
            text_content = f"""
            Registration Confirmation
            
            Dear {registration.full_name},
            
            Thank you for registering for our {registration.program} training program!
            
            Registration Details:
            - Registration ID: {registration.registration_code}
            - Program: {registration.program}
            - Email: {registration.email}
            - Phone: {registration.phone or 'Not provided'}
            - Experience Level: {registration.experience_level or 'Not specified'}
            
            What's Next?
            1. Our team will contact you within 24 hours
            2. We'll schedule a consultation call
            3. We'll create your personalized learning plan
            
            Need immediate assistance?
            WhatsApp: +254748370734
            Email: training@seovosolutions.com
            
            Best regards,
            Seovo Solutions Team
            """
            
            email = EmailMultiAlternatives(
                subject=subject,
                body=text_content,
                from_email=settings.DEFAULT_FROM_EMAIL,
                to=[registration.email],
                reply_to=[settings.CONTACT_EMAIL]
            )
            email.attach_alternative(html_content, "text/html")
            email.send(fail_silently=True)
            
            logger.info(f"📧 Confirmation email sent to {registration.email}")
            
        except Exception as e:
            logger.error(f"❌ Failed to send confirmation email: {e}")

# =========================================================================
# ENHANCED AI CHAT VIEW - PRODUCTION READY
# =========================================================================

class ProductionAIChatView(APIView):
    """Production-ready AI chat with rate limiting and comprehensive features"""
    throttle_classes = [AnonRateThrottle]
    parser_classes = [JSONParser]
    
    def post(self, request):
        """Handle AI chat requests"""
        start_time = time.time()
        
        try:
            # Validate input
            data = request.data or {}
            message = data.get('message', '').strip()
            
            if not message:
                return Response({
                    'error': 'Message is required'
                }, status=status.HTTP_400_BAD_REQUEST)
            
            if len(message) > 1000:
                return Response({
                    'error': 'Message too long (max 1000 characters)'
                }, status.HTTP_400_BAD_REQUEST)
            
            # Get AI response
            response, service_used = ai_manager.get_response(
                message,
                preferred_service=data.get('preferred_service')
            )
            
            # Log conversation
            conversation_id = self._log_conversation(
                user_id=data.get('user_id', 'anonymous'),
                session_id=data.get('session_id', ''),
                message=message,
                response=response,
                service_used=service_used,
                response_time=time.time() - start_time
            )
            
            # Return response
            return Response({
                'reply': response,
                'service_used': service_used,
                'conversation_id': conversation_id,
                'response_time': round(time.time() - start_time, 2),
                'timestamp': timezone.now().isoformat()
            })
            
        except Exception as e:
            logger.error(f"❌ AI Chat error: {e}")
            return Response({
                'error': 'Service temporarily unavailable',
                'fallback_response': 'Please contact us directly for assistance: +254748370734'
            }, status=status.HTTP_503_SERVICE_UNAVAILABLE)
    
    @safety.database_operation
    def _log_conversation(self, user_id, session_id, message, response, service_used, response_time):
        """Log conversation to database"""
        try:
            conv = ConversationLog.objects.create(
                user_identifier=user_id,
                session_id=session_id or f"sess_{int(time.time())}",
                user_message=message,
                bot_reply=response,
                intent='general',  # Basic intent for now
                confidence=0.9 if service_used != 'fallback' else 0.5,
                sources=[],
                metadata={
                    'service_used': service_used,
                    'response_time': response_time,
                    'char_count': len(message)
                },
                response_time=response_time
            )
            return conv.id
        except Exception as e:
            logger.error(f"❌ Failed to log conversation: {e}")
            return None

# =========================================================================
# PRODUCTION HEALTH AND METRICS ENDPOINTS
# =========================================================================

class ProductionHealthView(APIView):
    """Comprehensive health check endpoint"""
    permission_classes = [permissions.AllowAny]
    
    @method_decorator(cache_page(30))  # Cache for 30 seconds
    def get(self, request):
        """Get system health status"""
        health_status = {
            'status': 'healthy',
            'timestamp': timezone.now().isoformat(),
            'version': getattr(settings, 'API_VERSION', '1.0.0'),
            'environment': getattr(settings, 'DJANGO_ENV', 'production'),
            'services': {},
            'features': FEATURES
        }
        
        # Check database
        try:
            from django.db import connection
            with connection.cursor() as cursor:
                cursor.execute("SELECT 1")
            health_status['services']['database'] = 'healthy'
        except Exception as e:
            health_status['services']['database'] = 'unhealthy'
            health_status['status'] = 'degraded'
        
        # Check cache
        try:
            cache.set('health_check', 'ok', 10)
            if cache.get('health_check') == 'ok':
                health_status['services']['cache'] = 'healthy'
            else:
                health_status['services']['cache'] = 'unhealthy'
        except Exception:
            health_status['services']['cache'] = 'unhealthy'
        
        # Check AI services
        ai_stats = ai_manager.get_stats()
        health_status['services']['ai'] = {
            'available_services': len(ai_stats['available_services']),
            'success_rate': round(ai_stats['success_rate'], 2),
            'total_requests': ai_stats['total_requests']
        }
        
        return Response(health_status)

class ProductionMetricsView(APIView):
    """Production metrics endpoint"""
    permission_classes = [permissions.IsAuthenticatedOrReadOnly]
    
    def get(self, request):
        """Get system metrics"""
        cache_key = 'production_metrics'
        cached = cache.get(cache_key)
        
        if cached:
            return Response(cached)
        
        metrics = {
            'timestamp': timezone.now().isoformat(),
            'system': {},
            'business': {},
            'ai': ai_manager.get_stats()
        }
        
        # System metrics
        if MODELS_AVAILABLE:
            try:
                # Database counts
                metrics['system']['database_counts'] = {
                    'training_registrations': TrainingRegistration.objects.count(),
                    'training_programs': TrainingProgram.objects.filter(is_active=True).count(),
                    'conversations': ConversationLog.objects.count(),
                    'contacts': Contact.objects.count(),
                    'posts': Post.objects.filter(published=True).count(),
                }
                
                # Recent activity
                last_24h = timezone.now() - timedelta(hours=24)
                metrics['system']['recent_activity'] = {
                    'registrations_last_24h': TrainingRegistration.objects.filter(
                        created_at__gte=last_24h
                    ).count(),
                    'conversations_last_24h': ConversationLog.objects.filter(
                        created_at__gte=last_24h
                    ).count(),
                }
                
            except Exception as e:
                logger.error(f"❌ Metrics collection error: {e}")
                metrics['system']['error'] = str(e)
        
        # Business metrics (with fallbacks)
        metrics['business'] = {
            'experience': getattr(settings, 'CEO_EXPERIENCE_YEARS', 4),
            'projects': getattr(settings, 'PROJECTS_BASELINE', 200),
            'students': getattr(settings, 'STUDENTS_BASELINE', 600),
            'trainings': getattr(settings, 'TRAININGS_BASELINE', 200),
            'clients': getattr(settings, 'CLIENTS_BASELINE', 205),
        }
        
        cache.set(cache_key, metrics, 60)  # Cache for 1 minute
        return Response(metrics)

# =========================================================================
# OTHER PRODUCTION VIEWSETS
# =========================================================================

class ContactViewSet(ProductionViewSet):
    """Contact form submissions"""
    queryset = Contact.objects.all() if MODELS_AVAILABLE else []
    serializer_class = ContactSerializer
    permission_classes = [permissions.AllowAny]
    
    def perform_create(self, serializer):
        """Handle contact form submission"""
        instance = serializer.save()
        
        # Send notification email
        try:
            send_mail(
                subject=f'📬 New Contact: {instance.subject}',
                message=f"""
                New contact form submission:
                
                Name: {instance.name}
                Email: {instance.email}
                Phone: {instance.phone or 'Not provided'}
                Subject: {instance.subject}
                Message: {instance.message}
                
                Type: {instance.contact_type}
                Priority: {instance.priority}
                
                Received: {timezone.now().strftime("%Y-%m-%d %H:%M:%S")}
                """.strip(),
                from_email=settings.DEFAULT_FROM_EMAIL,
                recipient_list=[settings.CONTACT_EMAIL],
                fail_silently=True
            )
        except Exception as e:
            logger.error(f"❌ Failed to send contact notification: {e}")

class PostViewSet(ProductionViewSet):
    """Blog posts with caching"""
    queryset = Post.objects.filter(published=True) if MODELS_AVAILABLE else []
    serializer_class = PostSerializer
    permission_classes = [permissions.AllowAny]
    filter_backends = [filters.SearchFilter, filters.OrderingFilter]
    search_fields = ['title', 'excerpt', 'content', 'tags']
    ordering_fields = ['created_at', 'published_at', 'view_count']
    ordering = ['-published_at']
    
    @method_decorator(cache_page(60 * 10))  # Cache for 10 minutes
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

class EventViewSet(ProductionViewSet):
    """Events with caching"""
    queryset = Event.objects.filter(published=True) if MODELS_AVAILABLE else []
    serializer_class = EventSerializer
    permission_classes = [permissions.AllowAny]
    
    @action(detail=False, methods=['get'])
    def upcoming(self, request):
        """Get upcoming events"""
        cache_key = 'upcoming_events'
        cached = cache.get(cache_key)
        
        if cached:
            return Response(cached)
        
        now = timezone.now()
        events = self.get_queryset().filter(
            start_time__gte=now
        ).order_by('start_time')[:10]
        
        serializer = self.get_serializer(events, many=True)
        data = serializer.data
        
        cache.set(cache_key, data, 60 * 5)  # Cache for 5 minutes
        return Response(data)

# =========================================================================
# API ROOT AND INITIALIZATION
# =========================================================================

@api_view(['GET'])
@permission_classes([permissions.AllowAny])
def api_root(request):
    """API root endpoint"""
    base_url = request.build_absolute_uri('/')[:-1]
    
    return Response({
        'service': 'SeovoSolutions API',
        'version': getattr(settings, 'API_VERSION', '1.0.0'),
        'status': 'operational',
        'endpoints': {
            'training': f'{base_url}/api/training/',
            'training-programs': f'{base_url}/api/training-programs/',
            'ai-chat': f'{base_url}/api/ai-chat/',
            'contacts': f'{base_url}/api/contacts/',
            'posts': f'{base_url}/api/posts/',
            'events': f'{base_url}/api/events/',
            'testimonials': f'{base_url}/api/testimonials/',
            'health': f'{base_url}/api/health/',
            'metrics': f'{base_url}/api/metrics/',
        },
        'documentation': 'https://seovosolutions.com/docs/api',
        'support': {
            'email': 'support@seovosolutions.com',
            'whatsapp': '+254748370734'
        }
    })

# =========================================================================
# INITIALIZATION
# =========================================================================

def initialize_production_services():
    """Initialize production services"""
    logger.info("🚀 Initializing production services...")
    
    # Log startup
    safety.create_system_metric(
        'system_startup',
        metadata={
            'timestamp': timezone.now().isoformat(),
            'features': FEATURES,
            'ai_services': [s for s, config in AI_CONFIG.items() if config.get('enabled', False)]
        }
    )
    
    logger.info("✅ Production services initialized")

# Initialize on module load
initialize_production_services()

# =========================================================================
# EXPORT VIEWS
# =========================================================================

__all__ = [
    'TrainingProgramViewSet',
    'EnhancedTrainingRegistrationViewSet',
    'ProductionAIChatView',
    'ProductionHealthView',
    'ProductionMetricsView',
    'ContactViewSet',
    'PostViewSet',
    'EventViewSet',
    'api_root',
]